<?php
/**
 * GateCloud - Personel Yönetimi
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/Auth.php';
require_once __DIR__ . '/includes/HikvisionISAPI.php';

Auth::requireLogin();

define('PAGE_TITLE', 'Personel');

$db = getDB();
$companyId = Auth::companyId();

// AJAX işlemleri
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_SERVER['HTTP_X_REQUESTED_WITH'])) {
    header('Content-Type: application/json');
    
    $action = $_POST['action'] ?? '';
    $userId = (int)($_POST['user_id'] ?? 0);
    
    switch ($action) {
        case 'add':
        case 'edit':
            $data = [
                'employee_no' => trim($_POST['employee_no'] ?? ''),
                'first_name' => trim($_POST['first_name'] ?? ''),
                'last_name' => trim($_POST['last_name'] ?? ''),
                'department_id' => $_POST['department_id'] ?: null,
                'card_no' => trim($_POST['card_no'] ?? '') ?: null,
                'email' => trim($_POST['email'] ?? '') ?: null,
                'phone' => trim($_POST['phone'] ?? '') ?: null,
                'tc_no' => trim($_POST['tc_no'] ?? '') ?: null,
                'job_title' => trim($_POST['job_title'] ?? '') ?: null,
                'hire_date' => $_POST['hire_date'] ?: null,
                'valid_from' => $_POST['valid_from'] ?: null,
                'valid_until' => $_POST['valid_until'] ?: null,
            ];
            
            if (empty($data['employee_no']) || empty($data['first_name']) || empty($data['last_name'])) {
                jsonError('Sicil no, ad ve soyad zorunludur');
            }
            
            // Sicil no benzersizlik kontrolü
            $stmt = $db->prepare("SELECT id FROM users WHERE company_id = ? AND employee_no = ? AND id != ?");
            $stmt->execute([$companyId, $data['employee_no'], $userId]);
            if ($stmt->fetch()) {
                jsonError('Bu sicil numarası zaten kullanılıyor');
            }
            
            if ($action === 'add') {
                if (!Auth::checkLimit('users')) {
                    jsonError('Personel limitine ulaştınız. Paketinizi yükseltin.');
                }
                
                $stmt = $db->prepare("INSERT INTO users (company_id, employee_no, first_name, last_name, department_id, card_no, email, phone, tc_no, job_title, hire_date, valid_from, valid_until, sync_status, created_at) 
                                      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())");
                $stmt->execute([$companyId, $data['employee_no'], $data['first_name'], $data['last_name'], $data['department_id'], $data['card_no'], $data['email'], $data['phone'], $data['tc_no'], $data['job_title'], $data['hire_date'], $data['valid_from'], $data['valid_until']]);
                $userId = $db->lastInsertId();
                Auth::logActivity('user_add', 'user', $userId);
                jsonSuccess(['id' => $userId], 'Personel başarıyla eklendi');
            } else {
                $stmt = $db->prepare("UPDATE users SET employee_no = ?, first_name = ?, last_name = ?, department_id = ?, card_no = ?, email = ?, phone = ?, tc_no = ?, job_title = ?, hire_date = ?, valid_from = ?, valid_until = ?, sync_status = 'pending', updated_at = NOW() WHERE id = ? AND company_id = ?");
                $stmt->execute([$data['employee_no'], $data['first_name'], $data['last_name'], $data['department_id'], $data['card_no'], $data['email'], $data['phone'], $data['tc_no'], $data['job_title'], $data['hire_date'], $data['valid_from'], $data['valid_until'], $userId, $companyId]);
                Auth::logActivity('user_update', 'user', $userId);
                jsonSuccess(null, 'Personel başarıyla güncellendi');
            }
            break;
            
        case 'delete':
            $stmt = $db->prepare("UPDATE users SET is_active = 0, sync_status = 'pending' WHERE id = ? AND company_id = ?");
            $stmt->execute([$userId, $companyId]);
            Auth::logActivity('user_delete', 'user', $userId);
            jsonSuccess(null, 'Personel silindi');
            break;
            
        case 'sync_to_device':
            $deviceId = (int)$_POST['device_id'];
            
            // Kullanıcı bilgisi
            $stmt = $db->prepare("SELECT * FROM users WHERE id = ? AND company_id = ?");
            $stmt->execute([$userId, $companyId]);
            $user = $stmt->fetch();
            
            if (!$user) jsonError('Personel bulunamadı');
            
            // Cihaz bilgisi
            $stmt = $db->prepare("SELECT * FROM devices WHERE id = ? AND company_id = ?");
            $stmt->execute([$deviceId, $companyId]);
            $device = $stmt->fetch();
            
            if (!$device) jsonError('Cihaz bulunamadı');
            
            $isapi = new HikvisionISAPI($device['ip_address'], $device['port'], $device['username'], $device['password'], $device['use_https']);
            
            // Kullanıcıyı cihaza ekle
            $result = $isapi->addUser($user['employee_no'], $user['first_name'] . ' ' . $user['last_name'], [
                'validFrom' => $user['valid_from'] ? date('Y-m-d\TH:i:s', strtotime($user['valid_from'])) : null,
                'validUntil' => $user['valid_until'] ? date('Y-m-d\TH:i:s', strtotime($user['valid_until'])) : null
            ]);
            
            if ($result === false) {
                jsonError('Cihaza gönderilemedi: ' . $isapi->getLastError());
            }
            
            // Kart varsa ekle
            if ($user['card_no']) {
                $isapi->addCard($user['employee_no'], $user['card_no']);
            }
            
            // Erişim kaydı
            $stmt = $db->prepare("INSERT INTO user_device_access (user_id, device_id, is_active, sync_status) VALUES (?, ?, 1, 'synced') ON DUPLICATE KEY UPDATE sync_status = 'synced'");
            $stmt->execute([$userId, $deviceId]);
            
            // Kullanıcı sync durumu
            $stmt = $db->prepare("UPDATE users SET sync_status = 'synced', last_sync = NOW() WHERE id = ?");
            $stmt->execute([$userId]);
            
            jsonSuccess(null, 'Personel cihaza gönderildi');
            break;
            
        case 'sync_all':
            $deviceId = (int)$_POST['device_id'];
            
            $stmt = $db->prepare("SELECT * FROM devices WHERE id = ? AND company_id = ?");
            $stmt->execute([$deviceId, $companyId]);
            $device = $stmt->fetch();
            
            if (!$device) jsonError('Cihaz bulunamadı');
            
            $stmt = $db->prepare("SELECT * FROM users WHERE company_id = ? AND is_active = 1");
            $stmt->execute([$companyId]);
            $users = $stmt->fetchAll();
            
            $isapi = new HikvisionISAPI($device['ip_address'], $device['port'], $device['username'], $device['password'], $device['use_https']);
            
            $success = 0;
            $failed = 0;
            
            foreach ($users as $user) {
                $result = $isapi->addUser($user['employee_no'], $user['first_name'] . ' ' . $user['last_name']);
                if ($result !== false) {
                    if ($user['card_no']) {
                        $isapi->addCard($user['employee_no'], $user['card_no']);
                    }
                    $success++;
                } else {
                    $failed++;
                }
            }
            
            jsonSuccess(['success' => $success, 'failed' => $failed], "{$success} personel gönderildi, {$failed} başarısız");
            break;
            
        case 'upload_face':
            if (!isset($_FILES['face_image']) || $_FILES['face_image']['error'] !== UPLOAD_ERR_OK) {
                jsonError('Fotoğraf yüklenemedi');
            }
            
            $stmt = $db->prepare("SELECT * FROM users WHERE id = ? AND company_id = ?");
            $stmt->execute([$userId, $companyId]);
            $user = $stmt->fetch();
            
            if (!$user) jsonError('Personel bulunamadı');
            
            // Resmi kaydet
            $imageData = file_get_contents($_FILES['face_image']['tmp_name']);
            $base64 = base64_encode($imageData);
            
            // Veritabanına kaydet
            $stmt = $db->prepare("INSERT INTO faces (user_id, face_data, sync_status, created_at) VALUES (?, ?, 'pending', NOW()) ON DUPLICATE KEY UPDATE face_data = ?, sync_status = 'pending', updated_at = NOW()");
            $stmt->execute([$userId, $base64, $base64]);
            
            $stmt = $db->prepare("UPDATE users SET has_face = 1 WHERE id = ?");
            $stmt->execute([$userId]);
            
            jsonSuccess(null, 'Yüz fotoğrafı kaydedildi');
            break;
            
        case 'get':
            $stmt = $db->prepare("SELECT u.*, d.name as department_name FROM users u LEFT JOIN departments d ON u.department_id = d.id WHERE u.id = ? AND u.company_id = ?");
            $stmt->execute([$userId, $companyId]);
            $user = $stmt->fetch();
            
            if (!$user) jsonError('Personel bulunamadı', 404);
            
            // Yüz verisi
            $stmt = $db->prepare("SELECT face_data FROM faces WHERE user_id = ?");
            $stmt->execute([$userId]);
            $face = $stmt->fetch();
            $user['face_data'] = $face ? $face['face_data'] : null;
            
            // Atanmış cihazlar
            $stmt = $db->prepare("SELECT d.id, d.name FROM user_device_access uda JOIN devices d ON uda.device_id = d.id WHERE uda.user_id = ? AND uda.is_active = 1");
            $stmt->execute([$userId]);
            $user['assigned_devices'] = $stmt->fetchAll();
            
            jsonSuccess($user);
            break;
            
        default:
            jsonError('Geçersiz işlem');
    }
    exit;
}

// Filtreleme
$search = $_GET['search'] ?? '';
$departmentFilter = $_GET['department'] ?? '';
$statusFilter = $_GET['status'] ?? '';
$page = max(1, (int)($_GET['page'] ?? 1));
$perPage = 25;
$offset = ($page - 1) * $perPage;

// Sorgu oluştur
$where = ["u.company_id = ?", "u.is_active = 1"];
$params = [$companyId];

if ($search) {
    $where[] = "(u.first_name LIKE ? OR u.last_name LIKE ? OR u.employee_no LIKE ? OR u.card_no LIKE ?)";
    $searchParam = "%{$search}%";
    $params = array_merge($params, [$searchParam, $searchParam, $searchParam, $searchParam]);
}

if ($departmentFilter) {
    $where[] = "u.department_id = ?";
    $params[] = $departmentFilter;
}

if ($statusFilter === 'synced') {
    $where[] = "u.sync_status = 'synced'";
} elseif ($statusFilter === 'pending') {
    $where[] = "u.sync_status = 'pending'";
}

$whereClause = implode(' AND ', $where);

// Toplam kayıt
$stmt = $db->prepare("SELECT COUNT(*) FROM users u WHERE {$whereClause}");
$stmt->execute($params);
$totalRecords = $stmt->fetchColumn();
$totalPages = ceil($totalRecords / $perPage);

// Kullanıcı listesi
$stmt = $db->prepare("
    SELECT u.*, d.name as department_name,
           (SELECT COUNT(*) FROM user_device_access WHERE user_id = u.id AND is_active = 1) as device_count
    FROM users u
    LEFT JOIN departments d ON u.department_id = d.id
    WHERE {$whereClause}
    ORDER BY u.first_name, u.last_name
    LIMIT {$perPage} OFFSET {$offset}
");
$stmt->execute($params);
$users = $stmt->fetchAll();

// Departmanlar
$stmt = $db->prepare("SELECT * FROM departments WHERE company_id = ? AND is_active = 1 ORDER BY name");
$stmt->execute([$companyId]);
$departments = $stmt->fetchAll();

// Cihazlar
$stmt = $db->prepare("SELECT * FROM devices WHERE company_id = ? AND is_active = 1 ORDER BY name");
$stmt->execute([$companyId]);
$devices = $stmt->fetchAll();

require_once __DIR__ . '/includes/header.php';
?>

<!-- Header -->
<div class="flex flex-col lg:flex-row justify-between items-start lg:items-center gap-4 mb-6">
    <!-- Filters -->
    <form method="GET" class="flex flex-wrap gap-3">
        <div class="relative">
            <input type="text" name="search" value="<?= e($search) ?>" 
                   placeholder="Ara..." 
                   class="pl-10 pr-4 py-2 border rounded-lg w-64">
            <i class="fas fa-search absolute left-3 top-1/2 -translate-y-1/2 text-gray-400"></i>
        </div>
        
        <select name="department" class="px-4 py-2 border rounded-lg">
            <option value="">Tüm Departmanlar</option>
            <?php foreach ($departments as $dept): ?>
            <option value="<?= $dept['id'] ?>" <?= $departmentFilter == $dept['id'] ? 'selected' : '' ?>><?= e($dept['name']) ?></option>
            <?php endforeach; ?>
        </select>
        
        <select name="status" class="px-4 py-2 border rounded-lg">
            <option value="">Tüm Durumlar</option>
            <option value="synced" <?= $statusFilter == 'synced' ? 'selected' : '' ?>>Senkronize</option>
            <option value="pending" <?= $statusFilter == 'pending' ? 'selected' : '' ?>>Bekliyor</option>
        </select>
        
        <button type="submit" class="px-4 py-2 bg-gray-100 rounded-lg hover:bg-gray-200">
            <i class="fas fa-filter mr-1"></i> Filtrele
        </button>
    </form>
    
    <div class="flex gap-2">
        <div class="relative" x-data="{ open: false }">
            <button @click="open = !open" class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700">
                <i class="fas fa-sync mr-2"></i> Toplu Senkronize
            </button>
            <div x-show="open" @click.away="open = false" class="absolute right-0 mt-2 w-56 bg-white rounded-lg shadow-lg border py-1 z-10">
                <?php foreach ($devices as $device): ?>
                <button onclick="syncAllToDevice(<?= $device['id'] ?>)" class="w-full text-left px-4 py-2 hover:bg-gray-50">
                    <i class="fas fa-server mr-2 text-gray-500"></i> <?= e($device['name']) ?>
                </button>
                <?php endforeach; ?>
            </div>
        </div>
        
        <button onclick="openUserModal()" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
            <i class="fas fa-plus mr-2"></i> Yeni Personel
        </button>
    </div>
</div>

<!-- Stats -->
<div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
    <div class="card p-4">
        <p class="text-sm text-gray-500">Toplam Personel</p>
        <p class="text-2xl font-bold"><?= number_format($totalRecords) ?></p>
    </div>
    <div class="card p-4">
        <p class="text-sm text-gray-500">Yüzü Kayıtlı</p>
        <?php
        $stmt = $db->prepare("SELECT COUNT(*) FROM users WHERE company_id = ? AND has_face = 1 AND is_active = 1");
        $stmt->execute([$companyId]);
        ?>
        <p class="text-2xl font-bold text-green-600"><?= number_format($stmt->fetchColumn()) ?></p>
    </div>
    <div class="card p-4">
        <p class="text-sm text-gray-500">Kartı Olan</p>
        <?php
        $stmt = $db->prepare("SELECT COUNT(*) FROM users WHERE company_id = ? AND card_no IS NOT NULL AND is_active = 1");
        $stmt->execute([$companyId]);
        ?>
        <p class="text-2xl font-bold text-blue-600"><?= number_format($stmt->fetchColumn()) ?></p>
    </div>
    <div class="card p-4">
        <p class="text-sm text-gray-500">Senkronize Bekleyen</p>
        <?php
        $stmt = $db->prepare("SELECT COUNT(*) FROM users WHERE company_id = ? AND sync_status = 'pending' AND is_active = 1");
        $stmt->execute([$companyId]);
        ?>
        <p class="text-2xl font-bold text-yellow-600"><?= number_format($stmt->fetchColumn()) ?></p>
    </div>
</div>

<!-- Table -->
<div class="card overflow-hidden">
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Personel</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Sicil No</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Departman</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Kart No</th>
                    <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase">Kayıtlar</th>
                    <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase">Durum</th>
                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase">İşlemler</th>
                </tr>
            </thead>
            <tbody class="divide-y">
                <?php foreach ($users as $user): ?>
                <tr class="hover:bg-gray-50">
                    <td class="px-6 py-4">
                        <div class="flex items-center">
                            <div class="w-10 h-10 bg-gray-200 rounded-full flex items-center justify-center mr-3">
                                <?php if ($user['has_face']): ?>
                                <i class="fas fa-user-circle text-green-600 text-xl"></i>
                                <?php else: ?>
                                <span class="text-gray-500 font-medium"><?= strtoupper(substr($user['first_name'], 0, 1)) ?></span>
                                <?php endif; ?>
                            </div>
                            <div>
                                <p class="font-medium text-gray-800"><?= e($user['first_name'] . ' ' . $user['last_name']) ?></p>
                                <?php if ($user['job_title']): ?>
                                <p class="text-xs text-gray-500"><?= e($user['job_title']) ?></p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </td>
                    <td class="px-6 py-4 text-gray-600"><?= e($user['employee_no']) ?></td>
                    <td class="px-6 py-4 text-gray-600"><?= e($user['department_name'] ?? '-') ?></td>
                    <td class="px-6 py-4">
                        <?php if ($user['card_no']): ?>
                        <span class="font-mono text-sm"><?= e($user['card_no']) ?></span>
                        <?php else: ?>
                        <span class="text-gray-400">-</span>
                        <?php endif; ?>
                    </td>
                    <td class="px-6 py-4 text-center">
                        <div class="flex justify-center gap-2">
                            <?php if ($user['has_face']): ?>
                            <span class="w-6 h-6 bg-green-100 rounded-full flex items-center justify-center" title="Yüz">
                                <i class="fas fa-smile text-green-600 text-xs"></i>
                            </span>
                            <?php endif; ?>
                            <?php if ($user['has_fingerprint']): ?>
                            <span class="w-6 h-6 bg-blue-100 rounded-full flex items-center justify-center" title="Parmak İzi">
                                <i class="fas fa-fingerprint text-blue-600 text-xs"></i>
                            </span>
                            <?php endif; ?>
                            <?php if ($user['card_no']): ?>
                            <span class="w-6 h-6 bg-purple-100 rounded-full flex items-center justify-center" title="Kart">
                                <i class="fas fa-id-card text-purple-600 text-xs"></i>
                            </span>
                            <?php endif; ?>
                        </div>
                    </td>
                    <td class="px-6 py-4 text-center">
                        <?php if ($user['sync_status'] == 'synced'): ?>
                        <span class="badge badge-success">Senkronize</span>
                        <?php elseif ($user['sync_status'] == 'error'): ?>
                        <span class="badge badge-danger">Hata</span>
                        <?php else: ?>
                        <span class="badge badge-warning">Bekliyor</span>
                        <?php endif; ?>
                    </td>
                    <td class="px-6 py-4 text-right">
                        <div class="flex justify-end gap-2">
                            <button onclick="viewUser(<?= $user['id'] ?>)" class="p-2 text-gray-500 hover:text-blue-600 hover:bg-blue-50 rounded">
                                <i class="fas fa-eye"></i>
                            </button>
                            <button onclick="editUser(<?= $user['id'] ?>)" class="p-2 text-gray-500 hover:text-green-600 hover:bg-green-50 rounded">
                                <i class="fas fa-edit"></i>
                            </button>
                            <div class="relative" x-data="{ open: false }">
                                <button @click="open = !open" class="p-2 text-gray-500 hover:text-purple-600 hover:bg-purple-50 rounded">
                                    <i class="fas fa-sync"></i>
                                </button>
                                <div x-show="open" @click.away="open = false" class="absolute right-0 mt-1 w-48 bg-white rounded-lg shadow-lg border py-1 z-10">
                                    <?php foreach ($devices as $device): ?>
                                    <button onclick="syncToDevice(<?= $user['id'] ?>, <?= $device['id'] ?>)" class="w-full text-left px-4 py-2 hover:bg-gray-50 text-sm">
                                        <?= e($device['name']) ?>
                                    </button>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            <button onclick="deleteUser(<?= $user['id'] ?>)" class="p-2 text-gray-500 hover:text-red-600 hover:bg-red-50 rounded">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </td>
                </tr>
                <?php endforeach; ?>
                
                <?php if (empty($users)): ?>
                <tr>
                    <td colspan="7" class="px-6 py-12 text-center text-gray-500">
                        <i class="fas fa-users text-4xl mb-2"></i>
                        <p>Kayıt bulunamadı</p>
                    </td>
                </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Pagination -->
    <?php if ($totalPages > 1): ?>
    <div class="px-6 py-4 border-t flex justify-between items-center">
        <p class="text-sm text-gray-500">
            Toplam <?= number_format($totalRecords) ?> kayıt
        </p>
        <div class="flex gap-1">
            <?php for ($i = 1; $i <= min(5, $totalPages); $i++): ?>
            <a href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&department=<?= $departmentFilter ?>&status=<?= $statusFilter ?>" 
               class="px-3 py-1 rounded <?= $i == $page ? 'bg-blue-600 text-white' : 'hover:bg-gray-100' ?>">
                <?= $i ?>
            </a>
            <?php endfor; ?>
        </div>
    </div>
    <?php endif; ?>
</div>

<!-- User Modal -->
<div id="userModal" class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden items-center justify-center">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-2xl mx-4 max-h-[90vh] overflow-y-auto">
        <div class="p-6 border-b flex justify-between items-center">
            <h3 class="text-xl font-semibold" id="userModalTitle">Yeni Personel</h3>
            <button onclick="closeUserModal()" class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times text-xl"></i>
            </button>
        </div>
        
        <form id="userForm" class="p-6">
            <input type="hidden" name="action" id="userFormAction" value="add">
            <input type="hidden" name="user_id" id="userFormId" value="">
            
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Sicil No *</label>
                    <input type="text" name="employee_no" id="userEmployeeNo" required
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Departman</label>
                    <select name="department_id" id="userDepartment" class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                        <option value="">Seçiniz</option>
                        <?php foreach ($departments as $dept): ?>
                        <option value="<?= $dept['id'] ?>"><?= e($dept['name']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Ad *</label>
                    <input type="text" name="first_name" id="userFirstName" required
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Soyad *</label>
                    <input type="text" name="last_name" id="userLastName" required
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">TC Kimlik No</label>
                    <input type="text" name="tc_no" id="userTcNo" maxlength="11"
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Kart No</label>
                    <input type="text" name="card_no" id="userCardNo"
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">E-posta</label>
                    <input type="email" name="email" id="userEmail"
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Telefon</label>
                    <input type="tel" name="phone" id="userPhone"
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Ünvan</label>
                    <input type="text" name="job_title" id="userJobTitle"
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">İşe Başlama</label>
                    <input type="date" name="hire_date" id="userHireDate"
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Geçerlilik Başlangıç</label>
                    <input type="datetime-local" name="valid_from" id="userValidFrom"
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Geçerlilik Bitiş</label>
                    <input type="datetime-local" name="valid_until" id="userValidUntil"
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>
            </div>
            
            <div class="flex gap-3 mt-6">
                <button type="button" onclick="closeUserModal()" class="flex-1 px-4 py-2 border rounded-lg hover:bg-gray-50">
                    İptal
                </button>
                <button type="submit" class="flex-1 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                    <i class="fas fa-save mr-2"></i> Kaydet
                </button>
            </div>
        </form>
    </div>
</div>

<!-- View User Modal -->
<div id="viewUserModal" class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden items-center justify-center">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-lg mx-4 max-h-[90vh] overflow-y-auto">
        <div class="p-6 border-b flex justify-between items-center">
            <h3 class="text-xl font-semibold">Personel Detayı</h3>
            <button onclick="closeViewModal()" class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times text-xl"></i>
            </button>
        </div>
        <div id="viewUserContent" class="p-6">
            <!-- Content loaded via AJAX -->
        </div>
    </div>
</div>

<script>
function openUserModal(user = null) {
    document.getElementById('userModal').classList.remove('hidden');
    document.getElementById('userModal').classList.add('flex');
    
    if (user) {
        document.getElementById('userModalTitle').textContent = 'Personel Düzenle';
        document.getElementById('userFormAction').value = 'edit';
        document.getElementById('userFormId').value = user.id;
        document.getElementById('userEmployeeNo').value = user.employee_no;
        document.getElementById('userDepartment').value = user.department_id || '';
        document.getElementById('userFirstName').value = user.first_name;
        document.getElementById('userLastName').value = user.last_name;
        document.getElementById('userTcNo').value = user.tc_no || '';
        document.getElementById('userCardNo').value = user.card_no || '';
        document.getElementById('userEmail').value = user.email || '';
        document.getElementById('userPhone').value = user.phone || '';
        document.getElementById('userJobTitle').value = user.job_title || '';
        document.getElementById('userHireDate').value = user.hire_date || '';
        document.getElementById('userValidFrom').value = user.valid_from ? user.valid_from.replace(' ', 'T') : '';
        document.getElementById('userValidUntil').value = user.valid_until ? user.valid_until.replace(' ', 'T') : '';
    } else {
        document.getElementById('userModalTitle').textContent = 'Yeni Personel';
        document.getElementById('userFormAction').value = 'add';
        document.getElementById('userForm').reset();
        document.getElementById('userFormId').value = '';
    }
}

function closeUserModal() {
    document.getElementById('userModal').classList.add('hidden');
    document.getElementById('userModal').classList.remove('flex');
}

async function editUser(id) {
    const formData = new FormData();
    formData.append('action', 'get');
    formData.append('user_id', id);
    
    try {
        const response = await fetch('', { method: 'POST', body: formData, headers: { 'X-Requested-With': 'XMLHttpRequest' } });
        const result = await response.json();
        if (result.success) {
            openUserModal(result.data);
        } else {
            showToast(result.message, 'error');
        }
    } catch (e) {
        showToast('Bir hata oluştu', 'error');
    }
}

async function viewUser(id) {
    const formData = new FormData();
    formData.append('action', 'get');
    formData.append('user_id', id);
    
    try {
        const response = await fetch('', { method: 'POST', body: formData, headers: { 'X-Requested-With': 'XMLHttpRequest' } });
        const result = await response.json();
        if (result.success) {
            const user = result.data;
            document.getElementById('viewUserContent').innerHTML = `
                <div class="text-center mb-6">
                    ${user.face_data 
                        ? `<img src="data:image/jpeg;base64,${user.face_data}" class="w-32 h-32 rounded-full mx-auto object-cover">`
                        : `<div class="w-32 h-32 bg-gray-200 rounded-full flex items-center justify-center mx-auto"><i class="fas fa-user text-4xl text-gray-400"></i></div>`
                    }
                    <h3 class="text-xl font-semibold mt-3">${user.first_name} ${user.last_name}</h3>
                    <p class="text-gray-500">${user.job_title || ''}</p>
                </div>
                <div class="space-y-3">
                    <div class="flex justify-between py-2 border-b"><span class="text-gray-500">Sicil No</span><span class="font-medium">${user.employee_no}</span></div>
                    <div class="flex justify-between py-2 border-b"><span class="text-gray-500">Departman</span><span>${user.department_name || '-'}</span></div>
                    <div class="flex justify-between py-2 border-b"><span class="text-gray-500">Kart No</span><span class="font-mono">${user.card_no || '-'}</span></div>
                    <div class="flex justify-between py-2 border-b"><span class="text-gray-500">E-posta</span><span>${user.email || '-'}</span></div>
                    <div class="flex justify-between py-2 border-b"><span class="text-gray-500">Telefon</span><span>${user.phone || '-'}</span></div>
                </div>
                <div class="mt-4">
                    <p class="text-sm font-medium text-gray-700 mb-2">Atanmış Cihazlar</p>
                    <div class="flex flex-wrap gap-2">
                        ${user.assigned_devices.length > 0 
                            ? user.assigned_devices.map(d => `<span class="badge badge-info">${d.name}</span>`).join('')
                            : '<span class="text-gray-400">Henüz atanmamış</span>'
                        }
                    </div>
                </div>
            `;
            document.getElementById('viewUserModal').classList.remove('hidden');
            document.getElementById('viewUserModal').classList.add('flex');
        }
    } catch (e) {
        showToast('Bir hata oluştu', 'error');
    }
}

function closeViewModal() {
    document.getElementById('viewUserModal').classList.add('hidden');
    document.getElementById('viewUserModal').classList.remove('flex');
}

// Form submit
document.getElementById('userForm').addEventListener('submit', async function(e) {
    e.preventDefault();
    const formData = new FormData(this);
    
    try {
        const response = await fetch('', { method: 'POST', body: formData, headers: { 'X-Requested-With': 'XMLHttpRequest' } });
        const result = await response.json();
        
        if (result.success) {
            showToast(result.message, 'success');
            closeUserModal();
            setTimeout(() => location.reload(), 1000);
        } else {
            showToast(result.message, 'error');
        }
    } catch (e) {
        showToast('Bir hata oluştu', 'error');
    }
});

async function deleteUser(id) {
    if (!confirm('Bu personeli silmek istediğinize emin misiniz?')) return;
    
    const formData = new FormData();
    formData.append('action', 'delete');
    formData.append('user_id', id);
    
    try {
        const response = await fetch('', { method: 'POST', body: formData, headers: { 'X-Requested-With': 'XMLHttpRequest' } });
        const result = await response.json();
        if (result.success) {
            showToast(result.message, 'success');
            setTimeout(() => location.reload(), 1000);
        } else {
            showToast(result.message, 'error');
        }
    } catch (e) {
        showToast('Bir hata oluştu', 'error');
    }
}

async function syncToDevice(userId, deviceId) {
    const formData = new FormData();
    formData.append('action', 'sync_to_device');
    formData.append('user_id', userId);
    formData.append('device_id', deviceId);
    
    try {
        const response = await fetch('', { method: 'POST', body: formData, headers: { 'X-Requested-With': 'XMLHttpRequest' } });
        const result = await response.json();
        showToast(result.message, result.success ? 'success' : 'error');
    } catch (e) {
        showToast('Bir hata oluştu', 'error');
    }
}

async function syncAllToDevice(deviceId) {
    if (!confirm('Tüm personeli bu cihaza göndermek istediğinize emin misiniz?')) return;
    
    const formData = new FormData();
    formData.append('action', 'sync_all');
    formData.append('device_id', deviceId);
    
    showToast('Senkronizasyon başlatıldı...', 'info');
    
    try {
        const response = await fetch('', { method: 'POST', body: formData, headers: { 'X-Requested-With': 'XMLHttpRequest' } });
        const result = await response.json();
        showToast(result.message, result.success ? 'success' : 'error');
    } catch (e) {
        showToast('Bir hata oluştu', 'error');
    }
}

document.addEventListener('keydown', e => { if (e.key === 'Escape') { closeUserModal(); closeViewModal(); } });
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
