<?php
/**
 * GateCloud - Ayarlar
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/Auth.php';

Auth::requireLogin();

if (!Auth::hasRole(['owner', 'admin'])) {
    header('Location: /dashboard.php');
    exit;
}

define('PAGE_TITLE', 'Ayarlar');

$db = getDB();
$companyId = Auth::companyId();
$company = Auth::company();
$admin = Auth::user();

$message = '';
$error = '';
$tab = $_GET['tab'] ?? 'company';

// Form işlemleri
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'update_company':
            $stmt = $db->prepare("UPDATE companies SET name = ?, email = ?, phone = ?, address = ?, city = ?, tax_number = ?, tax_office = ? WHERE id = ?");
            $stmt->execute([
                trim($_POST['name']),
                trim($_POST['email']),
                trim($_POST['phone']),
                trim($_POST['address']),
                trim($_POST['city']),
                trim($_POST['tax_number']),
                trim($_POST['tax_office']),
                $companyId
            ]);
            Auth::logActivity('company_update', 'company', $companyId);
            $message = 'Şirket bilgileri güncellendi.';
            break;
            
        case 'update_password':
            $currentPass = $_POST['current_password'];
            $newPass = $_POST['new_password'];
            $confirmPass = $_POST['confirm_password'];
            
            if (!verifyPassword($currentPass, $admin['password'])) {
                $error = 'Mevcut şifre hatalı.';
            } elseif (strlen($newPass) < 6) {
                $error = 'Yeni şifre en az 6 karakter olmalı.';
            } elseif ($newPass !== $confirmPass) {
                $error = 'Şifreler eşleşmiyor.';
            } else {
                $stmt = $db->prepare("UPDATE admins SET password = ? WHERE id = ?");
                $stmt->execute([hashPassword($newPass), $admin['id']]);
                $message = 'Şifreniz güncellendi.';
            }
            break;
            
        case 'create_api_key':
            if (!Auth::hasFeature('api_access')) {
                $error = 'API erişimi paketinize dahil değil.';
            } else {
                $apiKey = generateAPIKey();
                $apiSecret = bin2hex(random_bytes(64));
                
                $stmt = $db->prepare("INSERT INTO api_keys (company_id, name, api_key, api_secret) VALUES (?, ?, ?, ?)");
                $stmt->execute([$companyId, trim($_POST['key_name']), $apiKey, $apiSecret]);
                
                Auth::logActivity('api_key_create', 'api_key', $db->lastInsertId());
                $message = "API anahtarı oluşturuldu. Secret: {$apiSecret} (Bu değeri bir daha göremezsiniz!)";
            }
            break;
            
        case 'delete_api_key':
            $stmt = $db->prepare("DELETE FROM api_keys WHERE id = ? AND company_id = ?");
            $stmt->execute([$_POST['key_id'], $companyId]);
            $message = 'API anahtarı silindi.';
            break;
            
        case 'add_location':
            $stmt = $db->prepare("INSERT INTO locations (company_id, name, address, city) VALUES (?, ?, ?, ?)");
            $stmt->execute([$companyId, trim($_POST['loc_name']), trim($_POST['loc_address']), trim($_POST['loc_city'])]);
            $message = 'Lokasyon eklendi.';
            break;
            
        case 'delete_location':
            // Cihaz kontrolü
            $stmt = $db->prepare("SELECT COUNT(*) FROM devices WHERE location_id = ?");
            $stmt->execute([$_POST['location_id']]);
            if ($stmt->fetchColumn() > 0) {
                $error = 'Bu lokasyona bağlı cihazlar var.';
            } else {
                $stmt = $db->prepare("DELETE FROM locations WHERE id = ? AND company_id = ?");
                $stmt->execute([$_POST['location_id'], $companyId]);
                $message = 'Lokasyon silindi.';
            }
            break;
    }
    
    // Şirket bilgilerini yenile
    $stmt = $db->prepare("SELECT * FROM companies WHERE id = ?");
    $stmt->execute([$companyId]);
    $company = $stmt->fetch();
}

// API anahtarları
$stmt = $db->prepare("SELECT * FROM api_keys WHERE company_id = ? ORDER BY created_at DESC");
$stmt->execute([$companyId]);
$apiKeys = $stmt->fetchAll();

// Lokasyonlar
$stmt = $db->prepare("SELECT l.*, (SELECT COUNT(*) FROM devices WHERE location_id = l.id) as device_count FROM locations l WHERE l.company_id = ? ORDER BY l.name");
$stmt->execute([$companyId]);
$locations = $stmt->fetchAll();

// Paket bilgisi
$stmt = $db->prepare("SELECT * FROM packages WHERE id = ?");
$stmt->execute([$company['package_id']]);
$package = $stmt->fetch();

require_once __DIR__ . '/includes/header.php';
?>

<?php if ($message): ?>
<div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg mb-6">
    <i class="fas fa-check-circle mr-2"></i> <?= e($message) ?>
</div>
<?php endif; ?>

<?php if ($error): ?>
<div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg mb-6">
    <i class="fas fa-exclamation-circle mr-2"></i> <?= e($error) ?>
</div>
<?php endif; ?>

<!-- Tabs -->
<div class="border-b mb-6">
    <nav class="flex gap-4">
        <a href="?tab=company" class="px-4 py-3 border-b-2 <?= $tab == 'company' ? 'border-blue-600 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700' ?>">
            <i class="fas fa-building mr-2"></i> Şirket Bilgileri
        </a>
        <a href="?tab=locations" class="px-4 py-3 border-b-2 <?= $tab == 'locations' ? 'border-blue-600 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700' ?>">
            <i class="fas fa-map-marker-alt mr-2"></i> Lokasyonlar
        </a>
        <a href="?tab=api" class="px-4 py-3 border-b-2 <?= $tab == 'api' ? 'border-blue-600 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700' ?>">
            <i class="fas fa-key mr-2"></i> API Anahtarları
        </a>
        <a href="?tab=subscription" class="px-4 py-3 border-b-2 <?= $tab == 'subscription' ? 'border-blue-600 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700' ?>">
            <i class="fas fa-credit-card mr-2"></i> Abonelik
        </a>
        <a href="?tab=security" class="px-4 py-3 border-b-2 <?= $tab == 'security' ? 'border-blue-600 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700' ?>">
            <i class="fas fa-shield-alt mr-2"></i> Güvenlik
        </a>
    </nav>
</div>

<?php if ($tab == 'company'): ?>
<!-- Şirket Bilgileri -->
<div class="card p-6 max-w-2xl">
    <h3 class="text-lg font-semibold mb-6">Şirket Bilgileri</h3>
    
    <form method="POST" class="space-y-4">
        <input type="hidden" name="action" value="update_company">
        
        <div>
            <label class="block text-sm font-medium text-gray-700 mb-1">Şirket Adı</label>
            <input type="text" name="name" value="<?= e($company['name']) ?>" required
                   class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
        </div>
        
        <div class="grid grid-cols-2 gap-4">
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">E-posta</label>
                <input type="email" name="email" value="<?= e($company['email']) ?>"
                       class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Telefon</label>
                <input type="tel" name="phone" value="<?= e($company['phone']) ?>"
                       class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
            </div>
        </div>
        
        <div>
            <label class="block text-sm font-medium text-gray-700 mb-1">Adres</label>
            <textarea name="address" rows="2" class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500"><?= e($company['address']) ?></textarea>
        </div>
        
        <div class="grid grid-cols-3 gap-4">
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Şehir</label>
                <input type="text" name="city" value="<?= e($company['city']) ?>"
                       class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Vergi No</label>
                <input type="text" name="tax_number" value="<?= e($company['tax_number']) ?>"
                       class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Vergi Dairesi</label>
                <input type="text" name="tax_office" value="<?= e($company['tax_office']) ?>"
                       class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
            </div>
        </div>
        
        <div class="pt-4">
            <button type="submit" class="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                <i class="fas fa-save mr-2"></i> Kaydet
            </button>
        </div>
    </form>
</div>

<?php elseif ($tab == 'locations'): ?>
<!-- Lokasyonlar -->
<div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
    <div class="card p-6">
        <h3 class="text-lg font-semibold mb-4">Yeni Lokasyon Ekle</h3>
        <form method="POST" class="space-y-4">
            <input type="hidden" name="action" value="add_location">
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Lokasyon Adı *</label>
                <input type="text" name="loc_name" required class="w-full px-4 py-2 border rounded-lg">
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Adres</label>
                <input type="text" name="loc_address" class="w-full px-4 py-2 border rounded-lg">
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Şehir</label>
                <input type="text" name="loc_city" class="w-full px-4 py-2 border rounded-lg">
            </div>
            <button type="submit" class="w-full px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                <i class="fas fa-plus mr-2"></i> Ekle
            </button>
        </form>
    </div>
    
    <div class="card overflow-hidden">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Lokasyon</th>
                    <th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase">Cihaz</th>
                    <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">İşlem</th>
                </tr>
            </thead>
            <tbody class="divide-y">
                <?php foreach ($locations as $loc): ?>
                <tr class="hover:bg-gray-50">
                    <td class="px-4 py-3">
                        <p class="font-medium"><?= e($loc['name']) ?></p>
                        <p class="text-xs text-gray-500"><?= e($loc['city']) ?></p>
                    </td>
                    <td class="px-4 py-3 text-center"><?= $loc['device_count'] ?></td>
                    <td class="px-4 py-3 text-right">
                        <form method="POST" class="inline" onsubmit="return confirm('Silmek istediğinize emin misiniz?')">
                            <input type="hidden" name="action" value="delete_location">
                            <input type="hidden" name="location_id" value="<?= $loc['id'] ?>">
                            <button type="submit" class="text-red-600 hover:text-red-800"><i class="fas fa-trash"></i></button>
                        </form>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<?php elseif ($tab == 'api'): ?>
<!-- API Anahtarları -->
<?php if (!Auth::hasFeature('api_access')): ?>
<div class="bg-yellow-50 border border-yellow-200 text-yellow-700 px-4 py-3 rounded-lg mb-6">
    <i class="fas fa-lock mr-2"></i> API erişimi Pro ve üstü paketlerde kullanılabilir.
</div>
<?php else: ?>
<div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
    <div class="card p-6">
        <h3 class="text-lg font-semibold mb-4">Yeni API Anahtarı</h3>
        <form method="POST" class="space-y-4">
            <input type="hidden" name="action" value="create_api_key">
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Anahtar Adı</label>
                <input type="text" name="key_name" required placeholder="Örn: PdksCloud Entegrasyonu"
                       class="w-full px-4 py-2 border rounded-lg">
            </div>
            <button type="submit" class="w-full px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                <i class="fas fa-key mr-2"></i> Oluştur
            </button>
        </form>
        
        <div class="mt-6 p-4 bg-gray-50 rounded-lg">
            <h4 class="font-medium mb-2">API Dokümantasyonu</h4>
            <p class="text-sm text-gray-600 mb-2">Base URL: <code class="bg-gray-200 px-2 py-1 rounded"><?= APP_URL ?>/api</code></p>
            <a href="/api/docs" class="text-blue-600 hover:underline text-sm">Dokümantasyonu Görüntüle →</a>
        </div>
    </div>
    
    <div class="card overflow-hidden">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Ad</th>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">API Key</th>
                    <th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase">Durum</th>
                    <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">İşlem</th>
                </tr>
            </thead>
            <tbody class="divide-y">
                <?php foreach ($apiKeys as $key): ?>
                <tr class="hover:bg-gray-50">
                    <td class="px-4 py-3 font-medium"><?= e($key['name']) ?></td>
                    <td class="px-4 py-3">
                        <code class="text-xs bg-gray-100 px-2 py-1 rounded"><?= substr($key['api_key'], 0, 16) ?>...</code>
                    </td>
                    <td class="px-4 py-3 text-center">
                        <span class="badge <?= $key['is_active'] ? 'badge-success' : 'badge-danger' ?>">
                            <?= $key['is_active'] ? 'Aktif' : 'Pasif' ?>
                        </span>
                    </td>
                    <td class="px-4 py-3 text-right">
                        <form method="POST" class="inline" onsubmit="return confirm('Bu anahtarı silmek istediğinize emin misiniz?')">
                            <input type="hidden" name="action" value="delete_api_key">
                            <input type="hidden" name="key_id" value="<?= $key['id'] ?>">
                            <button type="submit" class="text-red-600 hover:text-red-800"><i class="fas fa-trash"></i></button>
                        </form>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>
<?php endif; ?>

<?php elseif ($tab == 'subscription'): ?>
<!-- Abonelik -->
<div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
    <div class="card p-6">
        <h3 class="text-lg font-semibold mb-4">Mevcut Paket</h3>
        <div class="text-center py-4">
            <span class="inline-block px-4 py-2 bg-blue-100 text-blue-800 rounded-full text-lg font-semibold">
                <?= e($package['name']) ?>
            </span>
        </div>
        <div class="space-y-3 mt-4">
            <div class="flex justify-between"><span class="text-gray-500">Cihaz Limiti</span><span class="font-medium"><?= $package['max_devices'] == 999 ? 'Sınırsız' : $package['max_devices'] ?></span></div>
            <div class="flex justify-between"><span class="text-gray-500">Personel Limiti</span><span class="font-medium"><?= $package['max_users'] == 9999 ? 'Sınırsız' : $package['max_users'] ?></span></div>
            <div class="flex justify-between"><span class="text-gray-500">Yönetici Limiti</span><span class="font-medium"><?= $package['max_admins'] ?></span></div>
            <div class="flex justify-between"><span class="text-gray-500">Abonelik Sonu</span><span class="font-medium"><?= formatDate($company['subscription_end']) ?></span></div>
        </div>
    </div>
    
    <?php
    $stmt = $db->prepare("SELECT * FROM packages WHERE is_active = 1 ORDER BY monthly_price");
    $stmt->execute();
    $packages = $stmt->fetchAll();
    
    foreach ($packages as $pkg):
        if ($pkg['id'] == $company['package_id']) continue;
    ?>
    <div class="card p-6 border-2 <?= $pkg['code'] == 'pro' ? 'border-blue-500' : 'border-transparent' ?>">
        <h3 class="text-lg font-semibold mb-2"><?= e($pkg['name']) ?></h3>
        <p class="text-3xl font-bold text-gray-800"><?= formatMoney($pkg['monthly_price']) ?><span class="text-sm font-normal text-gray-500">/ay</span></p>
        
        <ul class="mt-4 space-y-2 text-sm">
            <li><i class="fas fa-check text-green-500 mr-2"></i> <?= $pkg['max_devices'] == 999 ? 'Sınırsız' : $pkg['max_devices'] ?> cihaz</li>
            <li><i class="fas fa-check text-green-500 mr-2"></i> <?= $pkg['max_users'] == 9999 ? 'Sınırsız' : $pkg['max_users'] ?> personel</li>
            <li><i class="fas fa-check text-green-500 mr-2"></i> <?= $pkg['max_admins'] ?> yönetici</li>
        </ul>
        
        <button class="w-full mt-4 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
            Yükselt
        </button>
    </div>
    <?php endforeach; ?>
</div>

<?php elseif ($tab == 'security'): ?>
<!-- Güvenlik -->
<div class="max-w-xl">
    <div class="card p-6">
        <h3 class="text-lg font-semibold mb-6">Şifre Değiştir</h3>
        <form method="POST" class="space-y-4">
            <input type="hidden" name="action" value="update_password">
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Mevcut Şifre</label>
                <input type="password" name="current_password" required class="w-full px-4 py-2 border rounded-lg">
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Yeni Şifre</label>
                <input type="password" name="new_password" required minlength="6" class="w-full px-4 py-2 border rounded-lg">
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Yeni Şifre (Tekrar)</label>
                <input type="password" name="confirm_password" required class="w-full px-4 py-2 border rounded-lg">
            </div>
            
            <button type="submit" class="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                <i class="fas fa-key mr-2"></i> Şifreyi Değiştir
            </button>
        </form>
    </div>
</div>
<?php endif; ?>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
