<?php
/**
 * GateCloud - Hikvision ISAPI Kütüphanesi
 * DS-K1T321MFWX ve diğer Hikvision Access Control cihazları için
 */

class HikvisionISAPI {
    private $ip;
    private $port;
    private $username;
    private $password;
    private $useHttps;
    private $timeout;
    private $lastError;
    private $lastResponse;
    private $lastHttpCode;
    
    public function __construct($ip, $port = 80, $username = 'admin', $password = '', $useHttps = false) {
        $this->ip = $ip;
        $this->port = $port;
        $this->username = $username;
        $this->password = $password;
        $this->useHttps = $useHttps;
        $this->timeout = defined('ISAPI_TIMEOUT') ? ISAPI_TIMEOUT : 30;
    }
    
    private function getBaseUrl() {
        $protocol = $this->useHttps ? 'https' : 'http';
        return "{$protocol}://{$this->ip}:{$this->port}";
    }
    
    private function request($method, $endpoint, $data = null, $contentType = 'application/xml') {
        $url = $this->getBaseUrl() . $endpoint;
        
        $ch = curl_init();
        
        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_HTTPAUTH => CURLAUTH_DIGEST,
            CURLOPT_USERPWD => "{$this->username}:{$this->password}",
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_HTTPHEADER => [
                "Content-Type: {$contentType}",
                "Accept: application/xml, application/json"
            ]
        ];
        
        switch (strtoupper($method)) {
            case 'POST':
                $options[CURLOPT_POST] = true;
                if ($data) $options[CURLOPT_POSTFIELDS] = $data;
                break;
            case 'PUT':
                $options[CURLOPT_CUSTOMREQUEST] = 'PUT';
                if ($data) $options[CURLOPT_POSTFIELDS] = $data;
                break;
            case 'DELETE':
                $options[CURLOPT_CUSTOMREQUEST] = 'DELETE';
                if ($data) $options[CURLOPT_POSTFIELDS] = $data;
                break;
        }
        
        curl_setopt_array($ch, $options);
        
        $response = curl_exec($ch);
        $this->lastHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        $this->lastResponse = $response;
        
        if ($error) {
            $this->lastError = "CURL Error: {$error}";
            return false;
        }
        
        if ($this->lastHttpCode >= 400) {
            $this->lastError = "HTTP Error: {$this->lastHttpCode}";
            return false;
        }
        
        return $this->parseResponse($response);
    }
    
    private function parseResponse($response) {
        if (empty($response)) return true;
        
        // JSON dene
        $json = json_decode($response, true);
        if ($json !== null) return $json;
        
        // XML dene
        libxml_use_internal_errors(true);
        $xml = simplexml_load_string($response);
        if ($xml !== false) {
            return json_decode(json_encode($xml), true);
        }
        
        return $response;
    }
    
    public function getLastError() { return $this->lastError; }
    public function getLastResponse() { return $this->lastResponse; }
    public function getLastHttpCode() { return $this->lastHttpCode; }
    
    // ==================== CİHAZ BİLGİLERİ ====================
    
    public function getDeviceInfo() {
        return $this->request('GET', '/ISAPI/System/deviceInfo');
    }
    
    public function getDeviceStatus() {
        return $this->request('GET', '/ISAPI/System/status');
    }
    
    public function getCapabilities() {
        return $this->request('GET', '/ISAPI/AccessControl/capabilities');
    }
    
    public function checkConnection() {
        $result = $this->getDeviceInfo();
        return $result !== false;
    }
    
    public function getDeviceTime() {
        return $this->request('GET', '/ISAPI/System/time');
    }
    
    public function setDeviceTime($datetime = null) {
        $datetime = $datetime ?? date('Y-m-d\TH:i:s');
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<Time>
    <timeMode>manual</timeMode>
    <localTime>' . $datetime . '</localTime>
    <timeZone>CST-3:00:00</timeZone>
</Time>';
        return $this->request('PUT', '/ISAPI/System/time', $xml);
    }
    
    // ==================== KULLANICI YÖNETİMİ ====================
    
    public function getUserCount() {
        $result = $this->request('GET', '/ISAPI/AccessControl/UserInfo/Count?format=json');
        return $result['UserInfoCount']['userNumber'] ?? 0;
    }
    
    public function searchUsers($position = 0, $maxResults = 30) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<UserInfoSearchCond>
    <searchID>' . uniqid() . '</searchID>
    <maxResults>' . $maxResults . '</maxResults>
    <searchResultPosition>' . $position . '</searchResultPosition>
</UserInfoSearchCond>';
        return $this->request('POST', '/ISAPI/AccessControl/UserInfo/Search?format=json', $xml);
    }
    
    public function getUser($employeeNo) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<UserInfoSearchCond>
    <searchID>' . uniqid() . '</searchID>
    <maxResults>1</maxResults>
    <searchResultPosition>0</searchResultPosition>
    <EmployeeNoList>
        <employeeNo>' . $employeeNo . '</employeeNo>
    </EmployeeNoList>
</UserInfoSearchCond>';
        $result = $this->request('POST', '/ISAPI/AccessControl/UserInfo/Search?format=json', $xml);
        if ($result && isset($result['UserInfoSearch']['UserInfo'])) {
            return $result['UserInfoSearch']['UserInfo'][0] ?? $result['UserInfoSearch']['UserInfo'];
        }
        return null;
    }
    
    public function addUser($employeeNo, $name, $options = []) {
        $validFrom = $options['validFrom'] ?? date('Y-m-d\T00:00:00');
        $validUntil = $options['validUntil'] ?? date('Y-m-d\T23:59:59', strtotime('+10 years'));
        $userType = $options['userType'] ?? 'normal';
        $doorRight = $options['doorRight'] ?? '1';
        
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<UserInfo>
    <employeeNo>' . $employeeNo . '</employeeNo>
    <name>' . htmlspecialchars($name) . '</name>
    <userType>' . $userType . '</userType>
    <Valid>
        <enable>true</enable>
        <beginTime>' . $validFrom . '</beginTime>
        <endTime>' . $validUntil . '</endTime>
    </Valid>
    <doorRight>' . $doorRight . '</doorRight>
    <RightPlan>
        <doorNo>1</doorNo>
        <planTemplateNo>1</planTemplateNo>
    </RightPlan>
</UserInfo>';
        return $this->request('POST', '/ISAPI/AccessControl/UserInfo/Record?format=json', $xml);
    }
    
    public function updateUser($employeeNo, $name, $options = []) {
        $validFrom = $options['validFrom'] ?? date('Y-m-d\T00:00:00');
        $validUntil = $options['validUntil'] ?? date('Y-m-d\T23:59:59', strtotime('+10 years'));
        $userType = $options['userType'] ?? 'normal';
        
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<UserInfo>
    <employeeNo>' . $employeeNo . '</employeeNo>
    <name>' . htmlspecialchars($name) . '</name>
    <userType>' . $userType . '</userType>
    <Valid>
        <enable>true</enable>
        <beginTime>' . $validFrom . '</beginTime>
        <endTime>' . $validUntil . '</endTime>
    </Valid>
</UserInfo>';
        return $this->request('PUT', '/ISAPI/AccessControl/UserInfo/Modify?format=json', $xml);
    }
    
    public function deleteUser($employeeNo) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<UserInfoDelCond>
    <EmployeeNoList>
        <employeeNo>' . $employeeNo . '</employeeNo>
    </EmployeeNoList>
</UserInfoDelCond>';
        return $this->request('PUT', '/ISAPI/AccessControl/UserInfo/Delete?format=json', $xml);
    }
    
    public function deleteAllUsers() {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<UserInfoDelCond>
    <DeleteAll>true</DeleteAll>
</UserInfoDelCond>';
        return $this->request('PUT', '/ISAPI/AccessControl/UserInfo/Delete?format=json', $xml);
    }
    
    // ==================== KART YÖNETİMİ ====================
    
    public function addCard($employeeNo, $cardNo, $cardType = 'normalCard') {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<CardInfo>
    <employeeNo>' . $employeeNo . '</employeeNo>
    <cardNo>' . $cardNo . '</cardNo>
    <cardType>' . $cardType . '</cardType>
</CardInfo>';
        return $this->request('POST', '/ISAPI/AccessControl/CardInfo/Record?format=json', $xml);
    }
    
    public function deleteCard($employeeNo, $cardNo = null) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<CardInfoDelCond>
    <EmployeeNoList>
        <employeeNo>' . $employeeNo . '</employeeNo>
    </EmployeeNoList>
</CardInfoDelCond>';
        return $this->request('PUT', '/ISAPI/AccessControl/CardInfo/Delete?format=json', $xml);
    }
    
    public function searchCards($position = 0, $maxResults = 30) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<CardInfoSearchCond>
    <searchID>' . uniqid() . '</searchID>
    <maxResults>' . $maxResults . '</maxResults>
    <searchResultPosition>' . $position . '</searchResultPosition>
</CardInfoSearchCond>';
        return $this->request('POST', '/ISAPI/AccessControl/CardInfo/Search?format=json', $xml);
    }
    
    // ==================== YÜZ TANIMA ====================
    
    public function addFace($employeeNo, $faceData) {
        // faceData base64 encoded JPEG olmalı
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<FaceInfo>
    <employeeNo>' . $employeeNo . '</employeeNo>
    <faceLibType>blackFD</faceLibType>
    <FDID>1</FDID>
    <FPID>' . $employeeNo . '</FPID>
</FaceInfo>';
        
        // Multipart request gerekli
        $boundary = uniqid();
        $body = "--{$boundary}\r\n";
        $body .= "Content-Disposition: form-data; name=\"FaceDataRecord\"\r\n";
        $body .= "Content-Type: application/xml\r\n\r\n";
        $body .= $xml . "\r\n";
        $body .= "--{$boundary}\r\n";
        $body .= "Content-Disposition: form-data; name=\"FaceImage\"; filename=\"face.jpg\"\r\n";
        $body .= "Content-Type: image/jpeg\r\n\r\n";
        $body .= base64_decode($faceData) . "\r\n";
        $body .= "--{$boundary}--\r\n";
        
        $url = $this->getBaseUrl() . '/ISAPI/Intelligent/FDLib/FaceDataRecord?format=json';
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_HTTPAUTH => CURLAUTH_DIGEST,
            CURLOPT_USERPWD => "{$this->username}:{$this->password}",
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $body,
            CURLOPT_HTTPHEADER => [
                "Content-Type: multipart/form-data; boundary={$boundary}"
            ]
        ]);
        
        $response = curl_exec($ch);
        $this->lastHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        $this->lastResponse = $response;
        return $this->parseResponse($response);
    }
    
    public function deleteFace($employeeNo) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<FaceInfoDelCond>
    <EmployeeNoList>
        <employeeNo>' . $employeeNo . '</employeeNo>
    </EmployeeNoList>
</FaceInfoDelCond>';
        return $this->request('PUT', '/ISAPI/Intelligent/FDLib/FDSearch/Delete?format=json&FDID=1', $xml);
    }
    
    public function getFaceCount() {
        $result = $this->request('GET', '/ISAPI/Intelligent/FDLib/Count?format=json&FDID=1');
        return $result['annotationNum'] ?? 0;
    }
    
    // ==================== PARMAK İZİ ====================
    
    public function addFingerprint($employeeNo, $fingerIndex, $fingerData) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<FingerPrintCfg>
    <employeeNo>' . $employeeNo . '</employeeNo>
    <fingerPrintID>' . $fingerIndex . '</fingerPrintID>
    <fingerType>normalFP</fingerType>
    <fingerData>' . $fingerData . '</fingerData>
</FingerPrintCfg>';
        return $this->request('POST', '/ISAPI/AccessControl/FingerPrint/SetUp?format=json', $xml);
    }
    
    public function deleteFingerprint($employeeNo, $fingerIndex = null) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<FingerPrintDelete>
    <employeeNo>' . $employeeNo . '</employeeNo>';
        if ($fingerIndex !== null) {
            $xml .= '<fingerPrintID>' . $fingerIndex . '</fingerPrintID>';
        }
        $xml .= '</FingerPrintDelete>';
        return $this->request('PUT', '/ISAPI/AccessControl/FingerPrint/Delete?format=json', $xml);
    }
    
    public function captureFingerprint() {
        return $this->request('POST', '/ISAPI/AccessControl/FingerPrint/Capture?format=json');
    }
    
    public function getFingerprints($employeeNo) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<FingerPrintCond>
    <searchID>' . uniqid() . '</searchID>
    <employeeNo>' . $employeeNo . '</employeeNo>
</FingerPrintCond>';
        return $this->request('POST', '/ISAPI/AccessControl/FingerPrintDownload?format=json', $xml);
    }
    
    // ==================== GEÇİŞ LOGLARI ====================
    
    public function getAccessLogs($startTime = null, $endTime = null, $position = 0, $maxResults = 30) {
        $startTime = $startTime ?? date('Y-m-d\T00:00:00', strtotime('-7 days'));
        $endTime = $endTime ?? date('Y-m-d\T23:59:59');
        
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<AcsEventCond>
    <searchID>' . uniqid() . '</searchID>
    <searchResultPosition>' . $position . '</searchResultPosition>
    <maxResults>' . $maxResults . '</maxResults>
    <major>0</major>
    <minor>0</minor>
    <startTime>' . $startTime . '</startTime>
    <endTime>' . $endTime . '</endTime>
</AcsEventCond>';
        return $this->request('POST', '/ISAPI/AccessControl/AcsEvent?format=json', $xml);
    }
    
    public function getLatestLogs($count = 10) {
        return $this->getAccessLogs(
            date('Y-m-d\T00:00:00', strtotime('-1 day')),
            date('Y-m-d\T23:59:59'),
            0,
            $count
        );
    }
    
    // ==================== KAPI KONTROLÜ ====================
    
    public function openDoor($doorNo = 1) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<RemoteControlDoor>
    <cmd>open</cmd>
</RemoteControlDoor>';
        return $this->request('PUT', '/ISAPI/AccessControl/RemoteControl/door/' . $doorNo, $xml);
    }
    
    public function closeDoor($doorNo = 1) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<RemoteControlDoor>
    <cmd>close</cmd>
</RemoteControlDoor>';
        return $this->request('PUT', '/ISAPI/AccessControl/RemoteControl/door/' . $doorNo, $xml);
    }
    
    public function keepDoorOpen($doorNo = 1) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<RemoteControlDoor>
    <cmd>alwaysOpen</cmd>
</RemoteControlDoor>';
        return $this->request('PUT', '/ISAPI/AccessControl/RemoteControl/door/' . $doorNo, $xml);
    }
    
    public function keepDoorClosed($doorNo = 1) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<RemoteControlDoor>
    <cmd>alwaysClose</cmd>
</RemoteControlDoor>';
        return $this->request('PUT', '/ISAPI/AccessControl/RemoteControl/door/' . $doorNo, $xml);
    }
    
    public function getDoorStatus($doorNo = 1) {
        return $this->request('GET', '/ISAPI/AccessControl/Door/status/' . $doorNo);
    }
    
    // ==================== ZAMAN PLANLARI ====================
    
    public function getTimePlans() {
        return $this->request('GET', '/ISAPI/AccessControl/UserRightWeekPlanCfg?format=json');
    }
    
    public function setTimePlan($planNo, $weekPlan) {
        // weekPlan: array of day configs
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<UserRightWeekPlanCfg>
    <weekPlanNo>' . $planNo . '</weekPlanNo>
    <enable>true</enable>';
        
        foreach ($weekPlan as $day => $times) {
            $xml .= '<WeekPlanCfg>
        <week>' . $day . '</week>
        <id>1</id>
        <enable>true</enable>
        <TimeSegment>
            <beginTime>' . $times['start'] . '</beginTime>
            <endTime>' . $times['end'] . '</endTime>
        </TimeSegment>
    </WeekPlanCfg>';
        }
        
        $xml .= '</UserRightWeekPlanCfg>';
        return $this->request('PUT', '/ISAPI/AccessControl/UserRightWeekPlanCfg/' . $planNo . '?format=json', $xml);
    }
    
    // ==================== ALARM YÖNETİMİ ====================
    
    public function getAlarmStatus() {
        return $this->request('GET', '/ISAPI/Event/notification/alertStream');
    }
    
    public function subscribeAlarms($callbackUrl) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<HttpHostNotification>
    <id>1</id>
    <url>' . htmlspecialchars($callbackUrl) . '</url>
    <protocolType>HTTP</protocolType>
    <parameterFormatType>XML</parameterFormatType>
    <addressingFormatType>ipaddress</addressingFormatType>
    <httpAuthenticationMethod>none</httpAuthenticationMethod>
</HttpHostNotification>';
        return $this->request('PUT', '/ISAPI/Event/notification/httpHosts/1', $xml);
    }
    
    // ==================== SİSTEM ====================
    
    public function reboot() {
        return $this->request('PUT', '/ISAPI/System/reboot');
    }
    
    public function factoryReset() {
        return $this->request('PUT', '/ISAPI/System/factoryReset');
    }
    
    public function exportConfig() {
        return $this->request('GET', '/ISAPI/System/configurationData');
    }
}
