<?php
/**
 * GateCloud - Authentication & Authorization
 * Multi-tenant oturum yönetimi
 */

class Auth {
    private static $admin = null;
    private static $company = null;
    private static $systemAdmin = null;
    
    /**
     * Oturumu başlat
     */
    public static function init() {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        
        // Oturum timeout kontrolü
        if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] > SESSION_TIMEOUT)) {
            self::logout();
            return;
        }
        
        $_SESSION['last_activity'] = time();
        
        // Admin oturumu varsa yükle
        if (isset($_SESSION['admin_id']) && isset($_SESSION['company_id'])) {
            self::loadAdmin($_SESSION['admin_id'], $_SESSION['company_id']);
        }
        
        // System admin oturumu varsa yükle
        if (isset($_SESSION['system_admin_id'])) {
            self::loadSystemAdmin($_SESSION['system_admin_id']);
        }
    }
    
    /**
     * Şirket kodu ile giriş
     */
    public static function loginWithCompanyCode($companyCode, $username, $password, $remember = false) {
        $db = getDB();
        
        // Şirketi bul
        $stmt = $db->prepare("SELECT * FROM companies WHERE code = ? AND is_active = 1");
        $stmt->execute([$companyCode]);
        $company = $stmt->fetch();
        
        if (!$company) {
            return ['success' => false, 'message' => 'Şirket bulunamadı veya aktif değil.'];
        }
        
        // Abonelik kontrolü
        if ($company['subscription_end'] && strtotime($company['subscription_end']) < time()) {
            return ['success' => false, 'message' => 'Şirket aboneliği sona ermiş.'];
        }
        
        // Admini bul
        $stmt = $db->prepare("SELECT * FROM admins WHERE company_id = ? AND username = ? AND is_active = 1");
        $stmt->execute([$company['id'], $username]);
        $admin = $stmt->fetch();
        
        if (!$admin || !password_verify($password, $admin['password'])) {
            // Başarısız giriş logla
            writeLog('warning', 'Failed login attempt', [
                'company_code' => $companyCode,
                'username' => $username,
                'ip' => getClientIP()
            ]);
            return ['success' => false, 'message' => 'Kullanıcı adı veya şifre hatalı.'];
        }
        
        // Oturumu başlat
        session_regenerate_id(true);
        $_SESSION['admin_id'] = $admin['id'];
        $_SESSION['company_id'] = $company['id'];
        $_SESSION['last_activity'] = time();
        
        // Remember me
        if ($remember) {
            $token = bin2hex(random_bytes(32));
            $expires = date('Y-m-d H:i:s', time() + REMEMBER_ME_DURATION);
            
            $stmt = $db->prepare("INSERT INTO sessions (id, company_id, admin_id, ip_address, user_agent, payload) 
                                  VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $token,
                $company['id'],
                $admin['id'],
                getClientIP(),
                $_SERVER['HTTP_USER_AGENT'] ?? '',
                json_encode(['expires' => $expires])
            ]);
            
            setcookie('remember_token', $token, time() + REMEMBER_ME_DURATION, '/', '', true, true);
        }
        
        // Son giriş güncelle
        $stmt = $db->prepare("UPDATE admins SET last_login = NOW() WHERE id = ?");
        $stmt->execute([$admin['id']]);
        
        // Aktivite logla
        self::logActivity('login', 'admin', $admin['id']);
        
        self::$admin = $admin;
        self::$company = $company;
        
        return ['success' => true, 'message' => 'Giriş başarılı.'];
    }
    
    /**
     * Sistem admin girişi
     */
    public static function loginSystemAdmin($username, $password) {
        $db = getDB();
        
        $stmt = $db->prepare("SELECT * FROM system_admins WHERE username = ? AND is_active = 1");
        $stmt->execute([$username]);
        $admin = $stmt->fetch();
        
        if (!$admin || !password_verify($password, $admin['password'])) {
            writeLog('warning', 'Failed system admin login', [
                'username' => $username,
                'ip' => getClientIP()
            ]);
            return ['success' => false, 'message' => 'Kullanıcı adı veya şifre hatalı.'];
        }
        
        session_regenerate_id(true);
        $_SESSION['system_admin_id'] = $admin['id'];
        $_SESSION['last_activity'] = time();
        
        $stmt = $db->prepare("UPDATE system_admins SET last_login = NOW() WHERE id = ?");
        $stmt->execute([$admin['id']]);
        
        self::$systemAdmin = $admin;
        
        return ['success' => true, 'message' => 'Giriş başarılı.'];
    }
    
    /**
     * Admin bilgilerini yükle
     */
    private static function loadAdmin($adminId, $companyId) {
        $db = getDB();
        
        $stmt = $db->prepare("SELECT a.*, c.name as company_name, c.code as company_code, 
                              c.package_id, c.subscription_end, c.settings as company_settings,
                              p.name as package_name, p.code as package_code, p.features as package_features,
                              p.max_devices, p.max_users, p.max_cards, p.max_admins
                              FROM admins a
                              JOIN companies c ON a.company_id = c.id
                              JOIN packages p ON c.package_id = p.id
                              WHERE a.id = ? AND a.company_id = ? AND a.is_active = 1 AND c.is_active = 1");
        $stmt->execute([$adminId, $companyId]);
        self::$admin = $stmt->fetch();
        
        if (self::$admin) {
            $stmt = $db->prepare("SELECT * FROM companies WHERE id = ?");
            $stmt->execute([$companyId]);
            self::$company = $stmt->fetch();
        }
    }
    
    /**
     * System admin bilgilerini yükle
     */
    private static function loadSystemAdmin($adminId) {
        $db = getDB();
        $stmt = $db->prepare("SELECT * FROM system_admins WHERE id = ? AND is_active = 1");
        $stmt->execute([$adminId]);
        self::$systemAdmin = $stmt->fetch();
    }
    
    /**
     * Çıkış yap
     */
    public static function logout() {
        if (isset($_COOKIE['remember_token'])) {
            $db = getDB();
            $stmt = $db->prepare("DELETE FROM sessions WHERE id = ?");
            $stmt->execute([$_COOKIE['remember_token']]);
            setcookie('remember_token', '', time() - 3600, '/', '', true, true);
        }
        
        $_SESSION = [];
        session_destroy();
        
        self::$admin = null;
        self::$company = null;
        self::$systemAdmin = null;
    }
    
    /**
     * Giriş yapılmış mı?
     */
    public static function check() {
        return self::$admin !== null;
    }
    
    /**
     * System admin mi?
     */
    public static function isSystemAdmin() {
        return self::$systemAdmin !== null;
    }
    
    /**
     * Aktif admin
     */
    public static function user() {
        return self::$admin;
    }
    
    /**
     * Aktif şirket
     */
    public static function company() {
        return self::$company;
    }
    
    /**
     * Aktif system admin
     */
    public static function systemUser() {
        return self::$systemAdmin;
    }
    
    /**
     * Company ID
     */
    public static function companyId() {
        return self::$company['id'] ?? null;
    }
    
    /**
     * Admin ID
     */
    public static function id() {
        return self::$admin['id'] ?? null;
    }
    
    /**
     * Yetki kontrolü
     */
    public static function can($permission) {
        if (!self::$admin) return false;
        
        // Owner her şeyi yapabilir
        if (self::$admin['role'] === 'owner') return true;
        
        // Admin çoğu şeyi yapabilir
        if (self::$admin['role'] === 'admin') {
            $restricted = ['company_settings', 'billing', 'delete_company'];
            return !in_array($permission, $restricted);
        }
        
        // Operator sadece izin verilenleri yapabilir
        $permissions = json_decode(self::$admin['permissions'] ?? '[]', true);
        return in_array($permission, $permissions);
    }
    
    /**
     * Rol kontrolü
     */
    public static function hasRole($role) {
        if (!self::$admin) return false;
        
        if (is_array($role)) {
            return in_array(self::$admin['role'], $role);
        }
        
        return self::$admin['role'] === $role;
    }
    
    /**
     * Paket özellik kontrolü
     */
    public static function hasFeature($feature) {
        if (!self::$admin) return false;
        
        $features = json_decode(self::$admin['package_features'] ?? '{}', true);
        return isset($features[$feature]) && $features[$feature] === true;
    }
    
    /**
     * Limit kontrolü
     */
    public static function checkLimit($type) {
        if (!self::$admin) return false;
        
        $db = getDB();
        $companyId = self::companyId();
        
        switch ($type) {
            case 'devices':
                $stmt = $db->prepare("SELECT COUNT(*) FROM devices WHERE company_id = ?");
                $stmt->execute([$companyId]);
                $count = $stmt->fetchColumn();
                return $count < self::$admin['max_devices'];
                
            case 'users':
                $stmt = $db->prepare("SELECT COUNT(*) FROM users WHERE company_id = ?");
                $stmt->execute([$companyId]);
                $count = $stmt->fetchColumn();
                return $count < self::$admin['max_users'];
                
            case 'admins':
                $stmt = $db->prepare("SELECT COUNT(*) FROM admins WHERE company_id = ?");
                $stmt->execute([$companyId]);
                $count = $stmt->fetchColumn();
                return $count < self::$admin['max_admins'];
        }
        
        return true;
    }
    
    /**
     * Aktivite logla
     */
    public static function logActivity($action, $entityType = null, $entityId = null, $oldValues = null, $newValues = null) {
        $db = getDB();
        
        $stmt = $db->prepare("INSERT INTO activity_logs 
            (company_id, admin_id, system_admin_id, action, entity_type, entity_id, old_values, new_values, ip_address, user_agent)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        
        $stmt->execute([
            self::companyId(),
            self::id(),
            self::$systemAdmin['id'] ?? null,
            $action,
            $entityType,
            $entityId,
            $oldValues ? json_encode($oldValues) : null,
            $newValues ? json_encode($newValues) : null,
            getClientIP(),
            $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
    }
    
    /**
     * Giriş gerekli middleware
     */
    public static function requireLogin() {
        self::init();
        
        if (!self::check()) {
            if (self::isAjax()) {
                jsonError('Oturum sonlandı. Lütfen tekrar giriş yapın.', 401);
            }
            header('Location: /login.php');
            exit;
        }
    }
    
    /**
     * System admin gerekli middleware
     */
    public static function requireSystemAdmin() {
        self::init();
        
        if (!self::isSystemAdmin()) {
            if (self::isAjax()) {
                jsonError('Yetkisiz erişim.', 403);
            }
            header('Location: /system/login.php');
            exit;
        }
    }
    
    /**
     * AJAX isteği mi?
     */
    private static function isAjax() {
        return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
               strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
    }
}
