<?php
/**
 * GateCloud - Cihaz Yönetimi
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/Auth.php';
require_once __DIR__ . '/includes/HikvisionISAPI.php';

Auth::requireLogin();

define('PAGE_TITLE', 'Cihazlar');

$db = getDB();
$companyId = Auth::companyId();

// AJAX işlemleri
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_SERVER['HTTP_X_REQUESTED_WITH'])) {
    header('Content-Type: application/json');
    
    $action = $_POST['action'] ?? '';
    $deviceId = (int)($_POST['device_id'] ?? 0);
    
    // Cihaz sahibi kontrolü
    if ($deviceId) {
        $stmt = $db->prepare("SELECT * FROM devices WHERE id = ? AND company_id = ?");
        $stmt->execute([$deviceId, $companyId]);
        $device = $stmt->fetch();
        if (!$device) {
            jsonError('Cihaz bulunamadı', 404);
        }
    }
    
    switch ($action) {
        case 'add':
        case 'edit':
            if (!Auth::can('manage_devices')) {
                jsonError('Bu işlem için yetkiniz yok', 403);
            }
            
            $data = [
                'name' => trim($_POST['name'] ?? ''),
                'ip_address' => trim($_POST['ip_address'] ?? ''),
                'port' => (int)($_POST['port'] ?? 80),
                'username' => trim($_POST['username'] ?? 'admin'),
                'password' => $_POST['password'] ?? '',
                'use_https' => isset($_POST['use_https']) ? 1 : 0,
                'direction' => $_POST['direction'] ?? 'both',
                'location_id' => $_POST['location_id'] ?: null
            ];
            
            if (empty($data['name']) || empty($data['ip_address'])) {
                jsonError('Cihaz adı ve IP adresi zorunludur');
            }
            
            if ($action === 'add') {
                if (!Auth::checkLimit('devices')) {
                    jsonError('Cihaz limitine ulaştınız. Paketinizi yükseltin.');
                }
                
                $stmt = $db->prepare("INSERT INTO devices (company_id, name, ip_address, port, username, password, use_https, direction, location_id, created_at) 
                                      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");
                $stmt->execute([$companyId, $data['name'], $data['ip_address'], $data['port'], $data['username'], $data['password'], $data['use_https'], $data['direction'], $data['location_id']]);
                $deviceId = $db->lastInsertId();
                Auth::logActivity('device_add', 'device', $deviceId);
                jsonSuccess(['id' => $deviceId], 'Cihaz başarıyla eklendi');
            } else {
                $sql = "UPDATE devices SET name = ?, ip_address = ?, port = ?, username = ?, use_https = ?, direction = ?, location_id = ?, updated_at = NOW()";
                $params = [$data['name'], $data['ip_address'], $data['port'], $data['username'], $data['use_https'], $data['direction'], $data['location_id']];
                
                if (!empty($data['password'])) {
                    $sql .= ", password = ?";
                    $params[] = $data['password'];
                }
                
                $sql .= " WHERE id = ? AND company_id = ?";
                $params[] = $deviceId;
                $params[] = $companyId;
                
                $stmt = $db->prepare($sql);
                $stmt->execute($params);
                Auth::logActivity('device_update', 'device', $deviceId);
                jsonSuccess(null, 'Cihaz başarıyla güncellendi');
            }
            break;
            
        case 'delete':
            if (!Auth::can('manage_devices')) {
                jsonError('Bu işlem için yetkiniz yok', 403);
            }
            
            $stmt = $db->prepare("UPDATE devices SET is_active = 0 WHERE id = ? AND company_id = ?");
            $stmt->execute([$deviceId, $companyId]);
            Auth::logActivity('device_delete', 'device', $deviceId);
            jsonSuccess(null, 'Cihaz silindi');
            break;
            
        case 'test_connection':
            $isapi = new HikvisionISAPI($device['ip_address'], $device['port'], $device['username'], $device['password'], $device['use_https']);
            
            if ($isapi->checkConnection()) {
                $info = $isapi->getDeviceInfo();
                $capabilities = $isapi->getCapabilities();
                
                $stmt = $db->prepare("UPDATE devices SET status = 'online', last_heartbeat = NOW(), 
                                      serial_number = ?, firmware_version = ?, model = ?, capabilities = ?, status_message = NULL WHERE id = ?");
                $stmt->execute([
                    $info['serialNumber'] ?? null,
                    $info['firmwareVersion'] ?? null,
                    $info['deviceName'] ?? $info['model'] ?? null,
                    json_encode($capabilities),
                    $deviceId
                ]);
                
                jsonSuccess([
                    'device_name' => $info['deviceName'] ?? '',
                    'serial' => $info['serialNumber'] ?? '',
                    'firmware' => $info['firmwareVersion'] ?? '',
                    'model' => $info['model'] ?? ''
                ], 'Bağlantı başarılı!');
            } else {
                $stmt = $db->prepare("UPDATE devices SET status = 'error', status_message = ? WHERE id = ?");
                $stmt->execute([$isapi->getLastError(), $deviceId]);
                jsonError('Bağlantı hatası: ' . $isapi->getLastError());
            }
            break;
            
        case 'open_door':
            $isapi = new HikvisionISAPI($device['ip_address'], $device['port'], $device['username'], $device['password'], $device['use_https']);
            if ($isapi->openDoor()) {
                Auth::logActivity('door_open', 'device', $deviceId);
                jsonSuccess(null, 'Kapı açıldı!');
            } else {
                jsonError('Kapı açılamadı: ' . $isapi->getLastError());
            }
            break;
            
        case 'close_door':
            $isapi = new HikvisionISAPI($device['ip_address'], $device['port'], $device['username'], $device['password'], $device['use_https']);
            if ($isapi->closeDoor()) {
                Auth::logActivity('door_close', 'device', $deviceId);
                jsonSuccess(null, 'Kapı kapatıldı!');
            } else {
                jsonError('Kapı kapatılamadı: ' . $isapi->getLastError());
            }
            break;
            
        case 'sync_time':
            $isapi = new HikvisionISAPI($device['ip_address'], $device['port'], $device['username'], $device['password'], $device['use_https']);
            if ($isapi->setDeviceTime()) {
                jsonSuccess(null, 'Cihaz saati senkronize edildi');
            } else {
                jsonError('Saat senkronize edilemedi: ' . $isapi->getLastError());
            }
            break;
            
        case 'get_stats':
            $isapi = new HikvisionISAPI($device['ip_address'], $device['port'], $device['username'], $device['password'], $device['use_https']);
            
            $userCount = $isapi->getUserCount();
            $faceCount = $isapi->getFaceCount();
            
            jsonSuccess([
                'user_count' => $userCount,
                'face_count' => $faceCount
            ]);
            break;
            
        case 'reboot':
            if (!Auth::hasRole(['owner', 'admin'])) {
                jsonError('Bu işlem için yetkiniz yok', 403);
            }
            
            $isapi = new HikvisionISAPI($device['ip_address'], $device['port'], $device['username'], $device['password'], $device['use_https']);
            if ($isapi->reboot()) {
                $stmt = $db->prepare("UPDATE devices SET status = 'offline' WHERE id = ?");
                $stmt->execute([$deviceId]);
                Auth::logActivity('device_reboot', 'device', $deviceId);
                jsonSuccess(null, 'Cihaz yeniden başlatılıyor...');
            } else {
                jsonError('Cihaz yeniden başlatılamadı: ' . $isapi->getLastError());
            }
            break;
            
        default:
            jsonError('Geçersiz işlem', 400);
    }
    exit;
}

// Cihaz listesi
$stmt = $db->prepare("
    SELECT d.*, l.name as location_name,
           (SELECT COUNT(*) FROM user_device_access WHERE device_id = d.id AND is_active = 1) as user_count,
           (SELECT COUNT(*) FROM access_logs WHERE device_id = d.id AND DATE(event_time) = CURDATE()) as today_logs
    FROM devices d
    LEFT JOIN locations l ON d.location_id = l.id
    WHERE d.company_id = ? AND d.is_active = 1
    ORDER BY d.name
");
$stmt->execute([$companyId]);
$devices = $stmt->fetchAll();

// Lokasyonlar
$stmt = $db->prepare("SELECT * FROM locations WHERE company_id = ? AND is_active = 1 ORDER BY name");
$stmt->execute([$companyId]);
$locations = $stmt->fetchAll();

// Paket limiti
$admin = Auth::user();
$deviceLimit = $admin['max_devices'];
$currentDeviceCount = count($devices);

require_once __DIR__ . '/includes/header.php';
?>

<!-- Header -->
<div class="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4 mb-6">
    <div>
        <p class="text-gray-500">
            <?= $currentDeviceCount ?> / <?= $deviceLimit == 999 ? '∞' : $deviceLimit ?> cihaz kullanılıyor
        </p>
    </div>
    <?php if ($currentDeviceCount < $deviceLimit): ?>
    <button onclick="openDeviceModal()" class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition">
        <i class="fas fa-plus mr-2"></i> Yeni Cihaz Ekle
    </button>
    <?php else: ?>
    <div class="text-yellow-600 bg-yellow-50 px-4 py-2 rounded-lg">
        <i class="fas fa-exclamation-triangle mr-2"></i> Cihaz limitine ulaştınız
    </div>
    <?php endif; ?>
</div>

<!-- Devices Grid -->
<div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
    <?php foreach ($devices as $device): ?>
    <div class="card p-6" id="device-<?= $device['id'] ?>">
        <!-- Header -->
        <div class="flex items-start justify-between mb-4">
            <div class="flex items-center">
                <div class="w-12 h-12 rounded-lg flex items-center justify-center mr-3
                    <?php if ($device['status'] == 'online'): ?>bg-green-100
                    <?php elseif ($device['status'] == 'error'): ?>bg-red-100
                    <?php else: ?>bg-gray-100<?php endif; ?>">
                    <i class="fas fa-door-open text-xl
                        <?php if ($device['status'] == 'online'): ?>text-green-600
                        <?php elseif ($device['status'] == 'error'): ?>text-red-600
                        <?php else: ?>text-gray-400<?php endif; ?>"></i>
                </div>
                <div>
                    <h3 class="font-semibold text-gray-800"><?= e($device['name']) ?></h3>
                    <p class="text-sm text-gray-500"><?= e($device['ip_address']) ?>:<?= $device['port'] ?></p>
                </div>
            </div>
            
            <!-- Dropdown Menu -->
            <div class="relative" x-data="{ open: false }">
                <button @click="open = !open" class="p-2 hover:bg-gray-100 rounded-lg">
                    <i class="fas fa-ellipsis-v text-gray-500"></i>
                </button>
                <div x-show="open" @click.away="open = false" 
                     class="absolute right-0 mt-2 w-48 bg-white rounded-lg shadow-lg border py-1 z-10">
                    <button onclick="editDevice(<?= $device['id'] ?>)" class="w-full text-left px-4 py-2 hover:bg-gray-50">
                        <i class="fas fa-edit mr-2 text-gray-500"></i> Düzenle
                    </button>
                    <button onclick="testConnection(<?= $device['id'] ?>)" class="w-full text-left px-4 py-2 hover:bg-gray-50">
                        <i class="fas fa-plug mr-2 text-gray-500"></i> Bağlantı Test
                    </button>
                    <button onclick="syncTime(<?= $device['id'] ?>)" class="w-full text-left px-4 py-2 hover:bg-gray-50">
                        <i class="fas fa-clock mr-2 text-gray-500"></i> Saat Senkronize
                    </button>
                    <hr class="my-1">
                    <button onclick="rebootDevice(<?= $device['id'] ?>)" class="w-full text-left px-4 py-2 hover:bg-gray-50 text-yellow-600">
                        <i class="fas fa-redo mr-2"></i> Yeniden Başlat
                    </button>
                    <button onclick="deleteDevice(<?= $device['id'] ?>)" class="w-full text-left px-4 py-2 hover:bg-gray-50 text-red-600">
                        <i class="fas fa-trash mr-2"></i> Sil
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Status Badge -->
        <div class="flex items-center gap-2 mb-4">
            <?php if ($device['status'] == 'online'): ?>
            <span class="badge badge-success"><i class="fas fa-check-circle mr-1"></i> Online</span>
            <?php elseif ($device['status'] == 'error'): ?>
            <span class="badge badge-danger"><i class="fas fa-exclamation-circle mr-1"></i> Hata</span>
            <?php else: ?>
            <span class="badge" style="background:#E5E7EB;color:#6B7280;"><i class="fas fa-circle mr-1"></i> Offline</span>
            <?php endif; ?>
            
            <?php if ($device['direction'] == 'in'): ?>
            <span class="badge badge-info">Giriş</span>
            <?php elseif ($device['direction'] == 'out'): ?>
            <span class="badge badge-warning">Çıkış</span>
            <?php endif; ?>
        </div>
        
        <!-- Info -->
        <div class="space-y-2 text-sm mb-4">
            <?php if ($device['location_name']): ?>
            <div class="flex justify-between">
                <span class="text-gray-500">Lokasyon:</span>
                <span class="text-gray-700"><?= e($device['location_name']) ?></span>
            </div>
            <?php endif; ?>
            
            <?php if ($device['model']): ?>
            <div class="flex justify-between">
                <span class="text-gray-500">Model:</span>
                <span class="text-gray-700"><?= e($device['model']) ?></span>
            </div>
            <?php endif; ?>
            
            <div class="flex justify-between">
                <span class="text-gray-500">Bugünkü Geçiş:</span>
                <span class="text-gray-700 font-medium"><?= number_format($device['today_logs']) ?></span>
            </div>
            
            <div class="flex justify-between">
                <span class="text-gray-500">Kayıtlı Kişi:</span>
                <span class="text-gray-700"><?= number_format($device['user_count']) ?></span>
            </div>
            
            <?php if ($device['last_heartbeat']): ?>
            <div class="flex justify-between">
                <span class="text-gray-500">Son İletişim:</span>
                <span class="text-gray-700"><?= formatDateTime($device['last_heartbeat'], 'H:i') ?></span>
            </div>
            <?php endif; ?>
        </div>
        
        <?php if ($device['status_message']): ?>
        <div class="text-xs text-red-600 bg-red-50 p-2 rounded mb-4">
            <?= e($device['status_message']) ?>
        </div>
        <?php endif; ?>
        
        <!-- Actions -->
        <div class="flex gap-2">
            <button onclick="openDoor(<?= $device['id'] ?>)" 
                    class="flex-1 bg-green-600 text-white py-2 px-3 rounded-lg hover:bg-green-700 transition text-sm"
                    <?= $device['status'] != 'online' ? 'disabled class="opacity-50"' : '' ?>>
                <i class="fas fa-door-open mr-1"></i> Kapıyı Aç
            </button>
            <button onclick="viewLogs(<?= $device['id'] ?>)"
                    class="flex-1 bg-gray-100 text-gray-700 py-2 px-3 rounded-lg hover:bg-gray-200 transition text-sm">
                <i class="fas fa-list mr-1"></i> Loglar
            </button>
        </div>
    </div>
    <?php endforeach; ?>
    
    <?php if (empty($devices)): ?>
    <div class="col-span-full text-center py-12">
        <div class="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
            <i class="fas fa-server text-3xl text-gray-400"></i>
        </div>
        <h3 class="text-lg font-medium text-gray-800 mb-2">Henüz cihaz eklenmemiş</h3>
        <p class="text-gray-500 mb-4">Hikvision cihazınızı ekleyerek başlayın</p>
        <button onclick="openDeviceModal()" class="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700">
            <i class="fas fa-plus mr-2"></i> İlk Cihazı Ekle
        </button>
    </div>
    <?php endif; ?>
</div>

<!-- Device Modal -->
<div id="deviceModal" class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden items-center justify-center">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-lg mx-4 max-h-[90vh] overflow-y-auto">
        <div class="p-6 border-b">
            <h3 class="text-xl font-semibold" id="modalTitle">Yeni Cihaz Ekle</h3>
        </div>
        
        <form id="deviceForm" class="p-6 space-y-4">
            <input type="hidden" name="action" id="formAction" value="add">
            <input type="hidden" name="device_id" id="deviceId" value="">
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Cihaz Adı *</label>
                <input type="text" name="name" id="deviceName" required
                       class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                       placeholder="Örn: Ana Giriş">
            </div>
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">IP Adresi *</label>
                    <input type="text" name="ip_address" id="deviceIP" required
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500"
                           placeholder="192.168.1.100">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Port</label>
                    <input type="number" name="port" id="devicePort" value="80"
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>
            </div>
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Kullanıcı Adı</label>
                    <input type="text" name="username" id="deviceUsername" value="admin"
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Şifre</label>
                    <input type="password" name="password" id="devicePassword"
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500"
                           placeholder="••••••••">
                </div>
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Lokasyon</label>
                <select name="location_id" id="deviceLocation"
                        class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                    <option value="">Seçiniz</option>
                    <?php foreach ($locations as $loc): ?>
                    <option value="<?= $loc['id'] ?>"><?= e($loc['name']) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Geçiş Yönü</label>
                <select name="direction" id="deviceDirection"
                        class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                    <option value="both">Giriş & Çıkış</option>
                    <option value="in">Sadece Giriş</option>
                    <option value="out">Sadece Çıkış</option>
                </select>
            </div>
            
            <div class="flex items-center">
                <input type="checkbox" name="use_https" id="deviceHttps" class="w-4 h-4 text-blue-600 rounded">
                <label for="deviceHttps" class="ml-2 text-sm text-gray-700">HTTPS Kullan</label>
            </div>
            
            <div class="flex gap-3 pt-4">
                <button type="button" onclick="closeDeviceModal()" 
                        class="flex-1 px-4 py-2 border rounded-lg hover:bg-gray-50">
                    İptal
                </button>
                <button type="submit" 
                        class="flex-1 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                    <i class="fas fa-save mr-2"></i> Kaydet
                </button>
            </div>
        </form>
    </div>
</div>

<script>
// Device data cache
const devicesData = <?= json_encode($devices) ?>;

function openDeviceModal(device = null) {
    document.getElementById('deviceModal').classList.remove('hidden');
    document.getElementById('deviceModal').classList.add('flex');
    
    if (device) {
        document.getElementById('modalTitle').textContent = 'Cihaz Düzenle';
        document.getElementById('formAction').value = 'edit';
        document.getElementById('deviceId').value = device.id;
        document.getElementById('deviceName').value = device.name;
        document.getElementById('deviceIP').value = device.ip_address;
        document.getElementById('devicePort').value = device.port;
        document.getElementById('deviceUsername').value = device.username;
        document.getElementById('devicePassword').value = '';
        document.getElementById('deviceLocation').value = device.location_id || '';
        document.getElementById('deviceDirection').value = device.direction;
        document.getElementById('deviceHttps').checked = device.use_https == 1;
    } else {
        document.getElementById('modalTitle').textContent = 'Yeni Cihaz Ekle';
        document.getElementById('formAction').value = 'add';
        document.getElementById('deviceForm').reset();
        document.getElementById('deviceId').value = '';
    }
}

function closeDeviceModal() {
    document.getElementById('deviceModal').classList.add('hidden');
    document.getElementById('deviceModal').classList.remove('flex');
}

function editDevice(id) {
    const device = devicesData.find(d => d.id == id);
    if (device) openDeviceModal(device);
}

// Form submit
document.getElementById('deviceForm').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    
    try {
        const response = await fetch('', {
            method: 'POST',
            body: formData,
            headers: { 'X-Requested-With': 'XMLHttpRequest' }
        });
        
        const result = await response.json();
        
        if (result.success) {
            showToast(result.message, 'success');
            closeDeviceModal();
            setTimeout(() => location.reload(), 1000);
        } else {
            showToast(result.message, 'error');
        }
    } catch (error) {
        showToast('Bir hata oluştu', 'error');
    }
});

async function deviceAction(action, deviceId, confirm_msg = null) {
    if (confirm_msg && !confirm(confirm_msg)) return;
    
    const formData = new FormData();
    formData.append('action', action);
    formData.append('device_id', deviceId);
    
    try {
        const response = await fetch('', {
            method: 'POST',
            body: formData,
            headers: { 'X-Requested-With': 'XMLHttpRequest' }
        });
        
        const result = await response.json();
        
        if (result.success) {
            showToast(result.message, 'success');
            if (['delete', 'test_connection'].includes(action)) {
                setTimeout(() => location.reload(), 1000);
            }
        } else {
            showToast(result.message, 'error');
        }
    } catch (error) {
        showToast('Bir hata oluştu', 'error');
    }
}

function testConnection(id) { deviceAction('test_connection', id); }
function openDoor(id) { deviceAction('open_door', id); }
function syncTime(id) { deviceAction('sync_time', id); }
function rebootDevice(id) { deviceAction('reboot', id, 'Cihazı yeniden başlatmak istediğinize emin misiniz?'); }
function deleteDevice(id) { deviceAction('delete', id, 'Bu cihazı silmek istediğinize emin misiniz?'); }

function viewLogs(deviceId) {
    window.location.href = '/access-logs.php?device_id=' + deviceId;
}

// Close modal on escape
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') closeDeviceModal();
});
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
