<?php
/**
 * GateCloud - Departman Yönetimi
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/Auth.php';

Auth::requireLogin();

if (!Auth::hasRole(['owner', 'admin'])) {
    header('Location: /dashboard.php');
    exit;
}

define('PAGE_TITLE', 'Departmanlar');

$db = getDB();
$companyId = Auth::companyId();

// AJAX işlemleri
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_SERVER['HTTP_X_REQUESTED_WITH'])) {
    header('Content-Type: application/json');
    
    $action = $_POST['action'] ?? '';
    $deptId = (int)($_POST['department_id'] ?? 0);
    
    switch ($action) {
        case 'add':
        case 'edit':
            $name = trim($_POST['name'] ?? '');
            $parentId = $_POST['parent_id'] ?: null;
            
            if (empty($name)) {
                jsonError('Departman adı zorunludur');
            }
            
            if ($action === 'add') {
                $stmt = $db->prepare("INSERT INTO departments (company_id, name, parent_id) VALUES (?, ?, ?)");
                $stmt->execute([$companyId, $name, $parentId]);
                Auth::logActivity('department_add', 'department', $db->lastInsertId());
                jsonSuccess(null, 'Departman eklendi');
            } else {
                $stmt = $db->prepare("UPDATE departments SET name = ?, parent_id = ? WHERE id = ? AND company_id = ?");
                $stmt->execute([$name, $parentId, $deptId, $companyId]);
                Auth::logActivity('department_update', 'department', $deptId);
                jsonSuccess(null, 'Departman güncellendi');
            }
            break;
            
        case 'delete':
            // Alt departman kontrolü
            $stmt = $db->prepare("SELECT COUNT(*) FROM departments WHERE parent_id = ? AND is_active = 1");
            $stmt->execute([$deptId]);
            if ($stmt->fetchColumn() > 0) {
                jsonError('Bu departmanın alt departmanları var. Önce onları silin.');
            }
            
            // Personel kontrolü
            $stmt = $db->prepare("SELECT COUNT(*) FROM users WHERE department_id = ? AND is_active = 1");
            $stmt->execute([$deptId]);
            if ($stmt->fetchColumn() > 0) {
                jsonError('Bu departmanda personel var. Önce personeli başka departmana taşıyın.');
            }
            
            $stmt = $db->prepare("UPDATE departments SET is_active = 0 WHERE id = ? AND company_id = ?");
            $stmt->execute([$deptId, $companyId]);
            Auth::logActivity('department_delete', 'department', $deptId);
            jsonSuccess(null, 'Departman silindi');
            break;
            
        default:
            jsonError('Geçersiz işlem');
    }
    exit;
}

// Departman listesi
$stmt = $db->prepare("
    SELECT d.*, 
           p.name as parent_name,
           (SELECT COUNT(*) FROM users WHERE department_id = d.id AND is_active = 1) as user_count
    FROM departments d
    LEFT JOIN departments p ON d.parent_id = p.id
    WHERE d.company_id = ? AND d.is_active = 1
    ORDER BY d.parent_id IS NULL DESC, d.name
");
$stmt->execute([$companyId]);
$departments = $stmt->fetchAll();

// Hiyerarşik yapı için
function buildTree($departments, $parentId = null) {
    $tree = [];
    foreach ($departments as $dept) {
        if ($dept['parent_id'] == $parentId) {
            $dept['children'] = buildTree($departments, $dept['id']);
            $tree[] = $dept;
        }
    }
    return $tree;
}

$tree = buildTree($departments);

require_once __DIR__ . '/includes/header.php';
?>

<div class="flex justify-between items-center mb-6">
    <p class="text-gray-500"><?= count($departments) ?> departman</p>
    <button onclick="openDeptModal()" class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700">
        <i class="fas fa-plus mr-2"></i> Yeni Departman
    </button>
</div>

<div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
    <!-- Tree View -->
    <div class="card p-6">
        <h3 class="text-lg font-semibold mb-4"><i class="fas fa-sitemap text-blue-600 mr-2"></i> Organizasyon Şeması</h3>
        
        <?php
        function renderTree($items, $level = 0) {
            if (empty($items)) return;
            echo '<ul class="' . ($level > 0 ? 'ml-6 mt-2' : '') . '">';
            foreach ($items as $item) {
                echo '<li class="mb-2">';
                echo '<div class="flex items-center justify-between p-3 rounded-lg hover:bg-gray-50 group">';
                echo '<div class="flex items-center">';
                if (!empty($item['children'])) {
                    echo '<i class="fas fa-folder-open text-yellow-500 mr-2"></i>';
                } else {
                    echo '<i class="fas fa-folder text-gray-400 mr-2"></i>';
                }
                echo '<span class="font-medium">' . e($item['name']) . '</span>';
                echo '<span class="ml-2 text-xs text-gray-400">(' . $item['user_count'] . ' kişi)</span>';
                echo '</div>';
                echo '<div class="hidden group-hover:flex gap-1">';
                echo '<button onclick="editDept(' . htmlspecialchars(json_encode($item)) . ')" class="p-1 text-blue-600 hover:bg-blue-50 rounded"><i class="fas fa-edit"></i></button>';
                echo '<button onclick="deleteDept(' . $item['id'] . ')" class="p-1 text-red-600 hover:bg-red-50 rounded"><i class="fas fa-trash"></i></button>';
                echo '</div>';
                echo '</div>';
                if (!empty($item['children'])) {
                    renderTree($item['children'], $level + 1);
                }
                echo '</li>';
            }
            echo '</ul>';
        }
        
        if (!empty($tree)) {
            renderTree($tree);
        } else {
            echo '<p class="text-gray-500 text-center py-8">Henüz departman eklenmemiş</p>';
        }
        ?>
    </div>
    
    <!-- Table View -->
    <div class="card overflow-hidden">
        <div class="p-4 border-b">
            <h3 class="font-semibold"><i class="fas fa-list text-blue-600 mr-2"></i> Liste Görünümü</h3>
        </div>
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Departman</th>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Üst Departman</th>
                    <th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase">Personel</th>
                    <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">İşlem</th>
                </tr>
            </thead>
            <tbody class="divide-y">
                <?php foreach ($departments as $dept): ?>
                <tr class="hover:bg-gray-50">
                    <td class="px-4 py-3 font-medium"><?= e($dept['name']) ?></td>
                    <td class="px-4 py-3 text-gray-600"><?= e($dept['parent_name'] ?? '-') ?></td>
                    <td class="px-4 py-3 text-center"><?= $dept['user_count'] ?></td>
                    <td class="px-4 py-3 text-right">
                        <button onclick="editDept(<?= htmlspecialchars(json_encode($dept)) ?>)" 
                                class="p-2 text-blue-600 hover:bg-blue-50 rounded">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button onclick="deleteDept(<?= $dept['id'] ?>)" 
                                class="p-2 text-red-600 hover:bg-red-50 rounded">
                            <i class="fas fa-trash"></i>
                        </button>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Modal -->
<div id="deptModal" class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden items-center justify-center">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-md mx-4">
        <div class="p-6 border-b">
            <h3 class="text-xl font-semibold" id="deptModalTitle">Yeni Departman</h3>
        </div>
        <form id="deptForm" class="p-6 space-y-4">
            <input type="hidden" name="action" id="deptAction" value="add">
            <input type="hidden" name="department_id" id="deptId" value="">
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Departman Adı *</label>
                <input type="text" name="name" id="deptName" required
                       class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Üst Departman</label>
                <select name="parent_id" id="deptParent" class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500">
                    <option value="">Ana Departman</option>
                    <?php foreach ($departments as $dept): ?>
                    <option value="<?= $dept['id'] ?>"><?= e($dept['name']) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="flex gap-3 pt-4">
                <button type="button" onclick="closeDeptModal()" class="flex-1 px-4 py-2 border rounded-lg hover:bg-gray-50">İptal</button>
                <button type="submit" class="flex-1 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">Kaydet</button>
            </div>
        </form>
    </div>
</div>

<script>
function openDeptModal(dept = null) {
    document.getElementById('deptModal').classList.remove('hidden');
    document.getElementById('deptModal').classList.add('flex');
    
    if (dept) {
        document.getElementById('deptModalTitle').textContent = 'Departman Düzenle';
        document.getElementById('deptAction').value = 'edit';
        document.getElementById('deptId').value = dept.id;
        document.getElementById('deptName').value = dept.name;
        document.getElementById('deptParent').value = dept.parent_id || '';
    } else {
        document.getElementById('deptModalTitle').textContent = 'Yeni Departman';
        document.getElementById('deptAction').value = 'add';
        document.getElementById('deptForm').reset();
        document.getElementById('deptId').value = '';
    }
}

function closeDeptModal() {
    document.getElementById('deptModal').classList.add('hidden');
    document.getElementById('deptModal').classList.remove('flex');
}

function editDept(dept) {
    openDeptModal(dept);
}

async function deleteDept(id) {
    if (!confirm('Bu departmanı silmek istediğinize emin misiniz?')) return;
    
    const formData = new FormData();
    formData.append('action', 'delete');
    formData.append('department_id', id);
    
    const response = await fetch('', { method: 'POST', body: formData, headers: { 'X-Requested-With': 'XMLHttpRequest' } });
    const result = await response.json();
    
    if (result.success) {
        showToast(result.message, 'success');
        setTimeout(() => location.reload(), 1000);
    } else {
        showToast(result.message, 'error');
    }
}

document.getElementById('deptForm').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const response = await fetch('', { method: 'POST', body: new FormData(this), headers: { 'X-Requested-With': 'XMLHttpRequest' } });
    const result = await response.json();
    
    if (result.success) {
        showToast(result.message, 'success');
        closeDeptModal();
        setTimeout(() => location.reload(), 1000);
    } else {
        showToast(result.message, 'error');
    }
});

document.addEventListener('keydown', e => { if (e.key === 'Escape') closeDeptModal(); });
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
