<?php
/**
 * GateCloud - Dashboard
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/Auth.php';

Auth::requireLogin();

define('PAGE_TITLE', 'Dashboard');

$db = getDB();
$companyId = Auth::companyId();

// İstatistikler
$stats = [];

// Toplam personel
$stmt = $db->prepare("SELECT COUNT(*) FROM users WHERE company_id = ? AND is_active = 1");
$stmt->execute([$companyId]);
$stats['total_users'] = $stmt->fetchColumn();

// Toplam cihaz
$stmt = $db->prepare("SELECT COUNT(*) FROM devices WHERE company_id = ? AND is_active = 1");
$stmt->execute([$companyId]);
$stats['total_devices'] = $stmt->fetchColumn();

// Online cihaz
$stmt = $db->prepare("SELECT COUNT(*) FROM devices WHERE company_id = ? AND status = 'online' AND is_active = 1");
$stmt->execute([$companyId]);
$stats['online_devices'] = $stmt->fetchColumn();

// Bugünkü geçişler
$stmt = $db->prepare("SELECT COUNT(*) FROM access_logs WHERE company_id = ? AND DATE(event_time) = CURDATE()");
$stmt->execute([$companyId]);
$stats['today_logs'] = $stmt->fetchColumn();

// Bugünkü benzersiz giriş
$stmt = $db->prepare("SELECT COUNT(DISTINCT user_id) FROM access_logs WHERE company_id = ? AND DATE(event_time) = CURDATE() AND direction = 'in'");
$stmt->execute([$companyId]);
$stats['today_entries'] = $stmt->fetchColumn();

// Son 7 günlük geçiş grafiği
$stmt = $db->prepare("
    SELECT DATE(event_time) as date, COUNT(*) as count 
    FROM access_logs 
    WHERE company_id = ? AND event_time >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
    GROUP BY DATE(event_time)
    ORDER BY date
");
$stmt->execute([$companyId]);
$weeklyData = $stmt->fetchAll();

$chartLabels = [];
$chartData = [];
for ($i = 6; $i >= 0; $i--) {
    $date = date('Y-m-d', strtotime("-{$i} days"));
    $chartLabels[] = date('d.m', strtotime($date));
    $found = false;
    foreach ($weeklyData as $row) {
        if ($row['date'] == $date) {
            $chartData[] = (int)$row['count'];
            $found = true;
            break;
        }
    }
    if (!$found) $chartData[] = 0;
}

// Son geçişler
$stmt = $db->prepare("
    SELECT al.*, u.first_name, u.last_name, u.employee_no, d.name as device_name
    FROM access_logs al
    LEFT JOIN users u ON al.user_id = u.id
    LEFT JOIN devices d ON al.device_id = d.id
    WHERE al.company_id = ?
    ORDER BY al.event_time DESC
    LIMIT 10
");
$stmt->execute([$companyId]);
$recentLogs = $stmt->fetchAll();

// Aktif alarmlar
$stmt = $db->prepare("
    SELECT a.*, d.name as device_name
    FROM alarms a
    LEFT JOIN devices d ON a.device_id = d.id
    WHERE a.company_id = ? AND a.acknowledged = 0
    ORDER BY a.event_time DESC
    LIMIT 5
");
$stmt->execute([$companyId]);
$activeAlarms = $stmt->fetchAll();

// Cihaz durumları
$stmt = $db->prepare("SELECT * FROM devices WHERE company_id = ? AND is_active = 1 ORDER BY name");
$stmt->execute([$companyId]);
$devices = $stmt->fetchAll();

require_once __DIR__ . '/includes/header.php';
?>

<!-- Stats Cards -->
<div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-6">
    <div class="card p-6">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-sm text-gray-500">Toplam Personel</p>
                <p class="text-3xl font-bold text-gray-800"><?= number_format($stats['total_users']) ?></p>
            </div>
            <div class="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
                <i class="fas fa-users text-blue-600 text-xl"></i>
            </div>
        </div>
    </div>
    
    <div class="card p-6">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-sm text-gray-500">Cihazlar</p>
                <p class="text-3xl font-bold text-gray-800">
                    <?= $stats['online_devices'] ?><span class="text-lg text-gray-400">/<?= $stats['total_devices'] ?></span>
                </p>
            </div>
            <div class="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center">
                <i class="fas fa-server text-green-600 text-xl"></i>
            </div>
        </div>
        <p class="text-xs text-green-600 mt-2">
            <i class="fas fa-circle text-xs mr-1"></i> <?= $stats['online_devices'] ?> online
        </p>
    </div>
    
    <div class="card p-6">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-sm text-gray-500">Bugünkü Geçişler</p>
                <p class="text-3xl font-bold text-gray-800"><?= number_format($stats['today_logs']) ?></p>
            </div>
            <div class="w-12 h-12 bg-purple-100 rounded-full flex items-center justify-center">
                <i class="fas fa-door-open text-purple-600 text-xl"></i>
            </div>
        </div>
    </div>
    
    <div class="card p-6">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-sm text-gray-500">Bugün Giriş Yapan</p>
                <p class="text-3xl font-bold text-gray-800"><?= number_format($stats['today_entries']) ?></p>
            </div>
            <div class="w-12 h-12 bg-yellow-100 rounded-full flex items-center justify-center">
                <i class="fas fa-sign-in-alt text-yellow-600 text-xl"></i>
            </div>
        </div>
    </div>
</div>

<div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
    <!-- Chart -->
    <div class="lg:col-span-2 card p-6">
        <h3 class="text-lg font-semibold text-gray-800 mb-4">
            <i class="fas fa-chart-line text-blue-600 mr-2"></i> Son 7 Gün Geçiş Grafiği
        </h3>
        <canvas id="weeklyChart" height="100"></canvas>
    </div>
    
    <!-- Device Status -->
    <div class="card p-6">
        <h3 class="text-lg font-semibold text-gray-800 mb-4">
            <i class="fas fa-server text-blue-600 mr-2"></i> Cihaz Durumları
        </h3>
        <div class="space-y-3">
            <?php foreach ($devices as $device): ?>
            <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                <div class="flex items-center">
                    <?php if ($device['status'] == 'online'): ?>
                    <span class="w-3 h-3 bg-green-500 rounded-full mr-3"></span>
                    <?php elseif ($device['status'] == 'error'): ?>
                    <span class="w-3 h-3 bg-red-500 rounded-full mr-3"></span>
                    <?php else: ?>
                    <span class="w-3 h-3 bg-gray-400 rounded-full mr-3"></span>
                    <?php endif; ?>
                    <div>
                        <p class="font-medium text-gray-800"><?= e($device['name']) ?></p>
                        <p class="text-xs text-gray-500"><?= e($device['ip_address']) ?></p>
                    </div>
                </div>
                <a href="/devices.php?id=<?= $device['id'] ?>" class="text-blue-600 hover:text-blue-800">
                    <i class="fas fa-external-link-alt"></i>
                </a>
            </div>
            <?php endforeach; ?>
            
            <?php if (empty($devices)): ?>
            <p class="text-gray-500 text-center py-4">Henüz cihaz eklenmemiş</p>
            <a href="/devices.php?action=add" class="block text-center text-blue-600 hover:underline">
                <i class="fas fa-plus mr-1"></i> Cihaz Ekle
            </a>
            <?php endif; ?>
        </div>
    </div>
</div>

<div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mt-6">
    <!-- Recent Logs -->
    <div class="card p-6">
        <div class="flex items-center justify-between mb-4">
            <h3 class="text-lg font-semibold text-gray-800">
                <i class="fas fa-history text-blue-600 mr-2"></i> Son Geçişler
            </h3>
            <a href="/access-logs.php" class="text-sm text-blue-600 hover:underline">Tümünü Gör</a>
        </div>
        <div class="overflow-x-auto">
            <table class="w-full text-sm">
                <thead>
                    <tr class="text-left text-gray-500 border-b">
                        <th class="pb-3">Personel</th>
                        <th class="pb-3">Cihaz</th>
                        <th class="pb-3">Zaman</th>
                        <th class="pb-3">Yön</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($recentLogs as $log): ?>
                    <tr class="border-b hover:bg-gray-50">
                        <td class="py-3">
                            <?php if ($log['first_name']): ?>
                            <?= e($log['first_name'] . ' ' . $log['last_name']) ?>
                            <span class="text-gray-400 text-xs">(<?= e($log['employee_no']) ?>)</span>
                            <?php else: ?>
                            <span class="text-gray-400">Bilinmiyor</span>
                            <?php endif; ?>
                        </td>
                        <td class="py-3 text-gray-600"><?= e($log['device_name']) ?></td>
                        <td class="py-3 text-gray-600"><?= formatDateTime($log['event_time'], 'H:i:s') ?></td>
                        <td class="py-3">
                            <?php if ($log['direction'] == 'in'): ?>
                            <span class="badge badge-success"><i class="fas fa-sign-in-alt mr-1"></i> Giriş</span>
                            <?php elseif ($log['direction'] == 'out'): ?>
                            <span class="badge badge-info"><i class="fas fa-sign-out-alt mr-1"></i> Çıkış</span>
                            <?php else: ?>
                            <span class="badge" style="background:#E5E7EB;color:#374151;">-</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                    
                    <?php if (empty($recentLogs)): ?>
                    <tr>
                        <td colspan="4" class="py-8 text-center text-gray-500">Henüz geçiş kaydı yok</td>
                    </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Active Alarms -->
    <div class="card p-6">
        <div class="flex items-center justify-between mb-4">
            <h3 class="text-lg font-semibold text-gray-800">
                <i class="fas fa-exclamation-triangle text-yellow-600 mr-2"></i> Aktif Alarmlar
            </h3>
            <a href="/alarms.php" class="text-sm text-blue-600 hover:underline">Tümünü Gör</a>
        </div>
        <div class="space-y-3">
            <?php foreach ($activeAlarms as $alarm): ?>
            <div class="p-4 rounded-lg <?= $alarm['severity'] == 'critical' ? 'bg-red-50 border border-red-200' : 'bg-yellow-50 border border-yellow-200' ?>">
                <div class="flex items-start justify-between">
                    <div>
                        <p class="font-medium <?= $alarm['severity'] == 'critical' ? 'text-red-800' : 'text-yellow-800' ?>">
                            <?php
                            $alarmTypes = [
                                'door_forced' => 'Kapı Zorlandı',
                                'door_open_timeout' => 'Kapı Açık Kaldı',
                                'tamper' => 'Cihaz Müdahalesi',
                                'offline' => 'Cihaz Çevrimdışı',
                                'auth_fail_multiple' => 'Çoklu Başarısız Giriş'
                            ];
                            echo $alarmTypes[$alarm['alarm_type']] ?? $alarm['alarm_type'];
                            ?>
                        </p>
                        <p class="text-sm text-gray-600"><?= e($alarm['device_name']) ?></p>
                        <p class="text-xs text-gray-500 mt-1"><?= formatDateTime($alarm['event_time']) ?></p>
                    </div>
                    <button onclick="acknowledgeAlarm(<?= $alarm['id'] ?>)" 
                            class="text-sm px-3 py-1 bg-white border rounded hover:bg-gray-50">
                        Onayla
                    </button>
                </div>
            </div>
            <?php endforeach; ?>
            
            <?php if (empty($activeAlarms)): ?>
            <div class="text-center py-8 text-gray-500">
                <i class="fas fa-check-circle text-4xl text-green-400 mb-2"></i>
                <p>Aktif alarm yok</p>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
// Weekly Chart
const ctx = document.getElementById('weeklyChart').getContext('2d');
new Chart(ctx, {
    type: 'line',
    data: {
        labels: <?= json_encode($chartLabels) ?>,
        datasets: [{
            label: 'Geçiş Sayısı',
            data: <?= json_encode($chartData) ?>,
            borderColor: '#3B82F6',
            backgroundColor: 'rgba(59, 130, 246, 0.1)',
            fill: true,
            tension: 0.4
        }]
    },
    options: {
        responsive: true,
        plugins: {
            legend: { display: false }
        },
        scales: {
            y: { beginAtZero: true }
        }
    }
});

// Acknowledge alarm
async function acknowledgeAlarm(id) {
    try {
        await apiRequest('/api/alarms/acknowledge.php', 'POST', { id });
        showToast('Alarm onaylandı');
        location.reload();
    } catch (e) {}
}
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
