<?php
/**
 * GateCloud - Cihazlardan Log Çekme Cron Job
 * Her 5 dakikada bir çalıştır: */5 * * * * php /path/to/sync_logs.php
 */

// CLI kontrolü
if (php_sapi_name() !== 'cli') {
    die('Bu script sadece CLI üzerinden çalıştırılabilir.');
}

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../includes/HikvisionISAPI.php';

echo "[" . date('Y-m-d H:i:s') . "] Log senkronizasyonu başlatılıyor...\n";

$db = getDB();

// Aktif cihazları al
$stmt = $db->prepare("
    SELECT d.*, c.id as company_id, c.name as company_name 
    FROM devices d
    JOIN companies c ON d.company_id = c.id
    WHERE d.is_active = 1 AND c.is_active = 1
");
$stmt->execute();
$devices = $stmt->fetchAll();

$totalLogs = 0;
$totalDevices = count($devices);
$successDevices = 0;

foreach ($devices as $device) {
    echo "  Cihaz: {$device['name']} ({$device['ip_address']})... ";
    
    try {
        $isapi = new HikvisionISAPI(
            $device['ip_address'],
            $device['port'],
            $device['username'],
            $device['password'],
            $device['use_https']
        );
        
        // Bağlantı kontrolü
        if (!$isapi->checkConnection()) {
            // Cihaz çevrimdışı
            $stmt = $db->prepare("UPDATE devices SET status = 'offline', status_message = ? WHERE id = ?");
            $stmt->execute([$isapi->getLastError(), $device['id']]);
            echo "OFFLINE\n";
            continue;
        }
        
        // Son senkronizasyon zamanı
        $lastSync = $device['last_sync'] ?? date('Y-m-d\TH:i:s', strtotime('-1 hour'));
        $lastSyncFormatted = date('Y-m-d\TH:i:s', strtotime($lastSync));
        $nowFormatted = date('Y-m-d\TH:i:s');
        
        // Logları çek
        $result = $isapi->getAccessLogs($lastSyncFormatted, $nowFormatted, 0, 100);
        
        if (!$result || !isset($result['AcsEvent'])) {
            echo "Log yok\n";
            $stmt = $db->prepare("UPDATE devices SET status = 'online', last_heartbeat = NOW() WHERE id = ?");
            $stmt->execute([$device['id']]);
            $successDevices++;
            continue;
        }
        
        $events = $result['AcsEvent']['InfoList'] ?? [];
        if (!is_array($events)) {
            $events = [$events];
        }
        
        $newLogs = 0;
        
        foreach ($events as $event) {
            // Employee no'yu al
            $employeeNo = $event['employeeNoString'] ?? $event['employeeNo'] ?? null;
            $cardNo = $event['cardNo'] ?? null;
            $eventTime = $event['time'] ?? null;
            
            if (!$eventTime) continue;
            
            // Daha önce kaydedilmiş mi kontrol et
            $stmt = $db->prepare("
                SELECT id FROM access_logs 
                WHERE device_id = ? AND event_time = ? AND employee_no = ?
            ");
            $stmt->execute([$device['id'], $eventTime, $employeeNo]);
            if ($stmt->fetch()) continue;
            
            // User ID bul
            $userId = null;
            if ($employeeNo) {
                $stmt = $db->prepare("SELECT id FROM users WHERE company_id = ? AND employee_no = ?");
                $stmt->execute([$device['company_id'], $employeeNo]);
                $user = $stmt->fetch();
                $userId = $user['id'] ?? null;
            }
            
            // Yön belirle
            $direction = 'unknown';
            $major = $event['major'] ?? 0;
            $minor = $event['minor'] ?? 0;
            
            // Hikvision event type'larına göre yön belirleme
            if ($device['direction'] === 'in') {
                $direction = 'in';
            } elseif ($device['direction'] === 'out') {
                $direction = 'out';
            } else {
                // Cihazın kendi verdiği yöne bak
                if (isset($event['attendanceStatus'])) {
                    $direction = $event['attendanceStatus'] === 'checkIn' ? 'in' : 'out';
                }
            }
            
            // Event type
            $eventType = 'card';
            if (isset($event['currentVerifyMode'])) {
                $verifyMode = strtolower($event['currentVerifyMode']);
                if (strpos($verifyMode, 'face') !== false) $eventType = 'face';
                elseif (strpos($verifyMode, 'finger') !== false) $eventType = 'fingerprint';
                elseif (strpos($verifyMode, 'card') !== false) $eventType = 'card';
            }
            
            // Status
            $status = 'success';
            if (isset($event['major']) && $event['major'] == 5) {
                $status = 'denied';
            }
            
            // Logu kaydet
            $stmt = $db->prepare("
                INSERT INTO access_logs (
                    company_id, device_id, user_id, employee_no, card_no, 
                    event_time, event_type, direction, verify_mode, door_no, status,
                    temperature, mask_status, raw_data, created_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            
            $stmt->execute([
                $device['company_id'],
                $device['id'],
                $userId,
                $employeeNo,
                $cardNo,
                date('Y-m-d H:i:s', strtotime($eventTime)),
                $eventType,
                $direction,
                $event['currentVerifyMode'] ?? null,
                $event['doorNo'] ?? 1,
                $status,
                $event['currTemperature'] ?? null,
                isset($event['mask']) ? ($event['mask'] == 1 ? 'wearing' : 'not_wearing') : 'unknown',
                json_encode($event)
            ]);
            
            $newLogs++;
        }
        
        // Cihaz durumunu güncelle
        $stmt = $db->prepare("
            UPDATE devices SET 
                status = 'online', 
                last_heartbeat = NOW(), 
                last_sync = NOW(),
                status_message = NULL
            WHERE id = ?
        ");
        $stmt->execute([$device['id']]);
        
        echo "{$newLogs} yeni log\n";
        $totalLogs += $newLogs;
        $successDevices++;
        
    } catch (Exception $e) {
        $stmt = $db->prepare("UPDATE devices SET status = 'error', status_message = ? WHERE id = ?");
        $stmt->execute([$e->getMessage(), $device['id']]);
        echo "HATA: {$e->getMessage()}\n";
    }
}

echo "[" . date('Y-m-d H:i:s') . "] Tamamlandı. ";
echo "{$successDevices}/{$totalDevices} cihaz, {$totalLogs} yeni log.\n";

// Log dosyasına yaz
writeLog('info', 'Log sync completed', [
    'total_devices' => $totalDevices,
    'success_devices' => $successDevices,
    'total_logs' => $totalLogs
]);
