<?php
/**
 * GateCloud - Geçiş Kontrol Sistemi
 * Konfigürasyon Dosyası
 */

// Hata raporlama (production'da kapatılmalı)
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Zaman dilimi
date_default_timezone_set('Europe/Istanbul');

// Oturum ayarları
ini_set('session.cookie_httponly', 1);
ini_set('session.use_strict_mode', 1);
ini_set('session.cookie_samesite', 'Strict');

// Veritabanı ayarları
define('DB_HOST', 'localhost');
define('DB_NAME', 'gatecloud');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_CHARSET', 'utf8mb4');

// Uygulama ayarları
define('APP_NAME', 'GateCloud');
define('APP_VERSION', '1.0.0');
define('APP_URL', 'https://gatecloud.biuygulama.com.tr');
define('APP_ENV', 'development'); // production, development

// Güvenlik
define('ENCRYPTION_KEY', 'CHANGE_THIS_TO_RANDOM_32_CHAR_STRING');
define('JWT_SECRET', 'CHANGE_THIS_TO_RANDOM_64_CHAR_STRING');
define('CSRF_TOKEN_NAME', 'gc_csrf_token');

// Dosya yükleme
define('UPLOAD_PATH', __DIR__ . '/../uploads/');
define('MAX_UPLOAD_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_IMAGE_TYPES', ['image/jpeg', 'image/png', 'image/gif']);

// Hikvision ISAPI ayarları
define('ISAPI_TIMEOUT', 30); // saniye
define('ISAPI_RETRY_COUNT', 3);
define('ISAPI_RETRY_DELAY', 1); // saniye

// Log ayarları
define('LOG_PATH', __DIR__ . '/../logs/');
define('LOG_LEVEL', 'debug'); // debug, info, warning, error

// Cache ayarları
define('CACHE_ENABLED', true);
define('CACHE_PATH', __DIR__ . '/../cache/');
define('CACHE_TTL', 3600); // 1 saat

// Email ayarları (SMTP)
define('SMTP_HOST', 'smtp.example.com');
define('SMTP_PORT', 587);
define('SMTP_USER', '');
define('SMTP_PASS', '');
define('SMTP_FROM', 'noreply@gatecloud.com.tr');
define('SMTP_FROM_NAME', 'GateCloud');

// PdksCloud entegrasyonu
define('PDKSCLOUD_API_URL', 'https://pdkscloud.biuygulama.com.tr/api');

// Rate limiting
define('RATE_LIMIT_REQUESTS', 100);
define('RATE_LIMIT_WINDOW', 60); // saniye

// Session timeout
define('SESSION_TIMEOUT', 3600); // 1 saat
define('REMEMBER_ME_DURATION', 30 * 24 * 3600); // 30 gün

// Sayfalama
define('DEFAULT_PAGE_SIZE', 25);
define('MAX_PAGE_SIZE', 100);

// Cron job token (güvenlik için)
define('CRON_TOKEN', 'CHANGE_THIS_TO_RANDOM_STRING');

/**
 * Veritabanı bağlantısı
 */
function getDB() {
    static $pdo = null;
    
    if ($pdo === null) {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci"
            ];
            $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            error_log("Database connection failed: " . $e->getMessage());
            die("Veritabanı bağlantısı kurulamadı.");
        }
    }
    
    return $pdo;
}

/**
 * Güvenli çıktı
 */
function e($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * CSRF token oluştur
 */
function generateCSRFToken() {
    if (empty($_SESSION[CSRF_TOKEN_NAME])) {
        $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
    }
    return $_SESSION[CSRF_TOKEN_NAME];
}

/**
 * CSRF token doğrula
 */
function validateCSRFToken($token) {
    return isset($_SESSION[CSRF_TOKEN_NAME]) && hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
}

/**
 * JSON yanıt
 */
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Başarılı JSON yanıt
 */
function jsonSuccess($data = null, $message = 'İşlem başarılı') {
    jsonResponse([
        'success' => true,
        'message' => $message,
        'data' => $data
    ]);
}

/**
 * Hatalı JSON yanıt
 */
function jsonError($message, $statusCode = 400, $errors = null) {
    jsonResponse([
        'success' => false,
        'message' => $message,
        'errors' => $errors
    ], $statusCode);
}

/**
 * UUID oluştur
 */
function generateUUID() {
    return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
        mt_rand(0, 0xffff), mt_rand(0, 0xffff),
        mt_rand(0, 0xffff),
        mt_rand(0, 0x0fff) | 0x4000,
        mt_rand(0, 0x3fff) | 0x8000,
        mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
    );
}

/**
 * API anahtarı oluştur
 */
function generateAPIKey() {
    return bin2hex(random_bytes(32));
}

/**
 * Log yaz
 */
function writeLog($level, $message, $context = []) {
    $levels = ['debug' => 0, 'info' => 1, 'warning' => 2, 'error' => 3];
    
    if ($levels[$level] < $levels[LOG_LEVEL]) {
        return;
    }
    
    $logFile = LOG_PATH . date('Y-m-d') . '.log';
    $timestamp = date('Y-m-d H:i:s');
    $contextStr = !empty($context) ? ' ' . json_encode($context, JSON_UNESCAPED_UNICODE) : '';
    $logLine = "[$timestamp] [$level] $message$contextStr" . PHP_EOL;
    
    file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);
}

/**
 * Şifre hash'le
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]);
}

/**
 * Şifre doğrula
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Tarih formatla
 */
function formatDate($date, $format = 'd.m.Y') {
    if (empty($date)) return '-';
    return date($format, strtotime($date));
}

/**
 * Tarih ve saat formatla
 */
function formatDateTime($datetime, $format = 'd.m.Y H:i') {
    if (empty($datetime)) return '-';
    return date($format, strtotime($datetime));
}

/**
 * Para formatla
 */
function formatMoney($amount, $currency = '₺') {
    return number_format($amount, 2, ',', '.') . ' ' . $currency;
}

/**
 * Dosya boyutu formatla
 */
function formatFileSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $i = 0;
    while ($bytes >= 1024 && $i < count($units) - 1) {
        $bytes /= 1024;
        $i++;
    }
    return round($bytes, 2) . ' ' . $units[$i];
}

/**
 * Slug oluştur
 */
function createSlug($text) {
    $text = mb_strtolower($text, 'UTF-8');
    $text = str_replace(['ı', 'ğ', 'ü', 'ş', 'ö', 'ç'], ['i', 'g', 'u', 's', 'o', 'c'], $text);
    $text = preg_replace('/[^a-z0-9]+/', '-', $text);
    return trim($text, '-');
}

/**
 * IP adresi al
 */
function getClientIP() {
    $headers = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
    foreach ($headers as $header) {
        if (!empty($_SERVER[$header])) {
            $ip = $_SERVER[$header];
            if (strpos($ip, ',') !== false) {
                $ip = trim(explode(',', $ip)[0]);
            }
            if (filter_var($ip, FILTER_VALIDATE_IP)) {
                return $ip;
            }
        }
    }
    return '0.0.0.0';
}

// Klasörleri oluştur
$directories = [UPLOAD_PATH, LOG_PATH, CACHE_PATH];
foreach ($directories as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}
