<?php
/**
 * GateCloud - API Endpoint: Access Logs
 * PdksCloud entegrasyonu için
 */

require_once __DIR__ . '/../../config/config.php';

header('Content-Type: application/json; charset=utf-8');

// CORS
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-API-Key, X-Timestamp, X-Signature');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit;
}

// API key doğrulama
function authenticateAPI() {
    $apiKey = $_SERVER['HTTP_X_API_KEY'] ?? '';
    $timestamp = $_SERVER['HTTP_X_TIMESTAMP'] ?? '';
    $signature = $_SERVER['HTTP_X_SIGNATURE'] ?? '';
    
    if (empty($apiKey)) {
        jsonError('API key gerekli', 401);
    }
    
    // Timestamp kontrolü (5 dakika tolerans)
    if (abs(time() - (int)$timestamp) > 300) {
        jsonError('İstek süresi doldu', 401);
    }
    
    $db = getDB();
    $stmt = $db->prepare("
        SELECT ak.*, c.id as company_id, c.is_active as company_active
        FROM api_keys ak
        JOIN companies c ON ak.company_id = c.id
        WHERE ak.api_key = ? AND ak.is_active = 1
    ");
    $stmt->execute([$apiKey]);
    $key = $stmt->fetch();
    
    if (!$key) {
        jsonError('Geçersiz API key', 401);
    }
    
    if (!$key['company_active']) {
        jsonError('Şirket hesabı aktif değil', 403);
    }
    
    // Expiry kontrolü
    if ($key['expires_at'] && strtotime($key['expires_at']) < time()) {
        jsonError('API key süresi dolmuş', 401);
    }
    
    // İmza doğrulama
    $method = $_SERVER['REQUEST_METHOD'];
    $endpoint = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    $body = file_get_contents('php://input');
    
    $signatureData = $method . $endpoint . $timestamp . $body;
    $expectedSignature = hash_hmac('sha256', $signatureData, $key['api_secret']);
    
    if (!hash_equals($expectedSignature, $signature)) {
        jsonError('Geçersiz imza', 401);
    }
    
    // Son kullanım güncelle
    $stmt = $db->prepare("UPDATE api_keys SET last_used = NOW() WHERE id = ?");
    $stmt->execute([$key['id']]);
    
    return $key;
}

// Rate limiting
function checkRateLimit($apiKeyId, $limit = 1000) {
    $db = getDB();
    
    $stmt = $db->prepare("
        SELECT COUNT(*) as count FROM activity_logs 
        WHERE action = 'api_request' AND entity_id = ? 
        AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)
    ");
    $stmt->execute([$apiKeyId]);
    $result = $stmt->fetch();
    
    if ($result['count'] >= $limit) {
        jsonError('Rate limit aşıldı', 429);
    }
    
    // Log
    $stmt = $db->prepare("
        INSERT INTO activity_logs (action, entity_type, entity_id, ip_address, created_at)
        VALUES ('api_request', 'api_key', ?, ?, NOW())
    ");
    $stmt->execute([$apiKeyId, getClientIP()]);
}

// Authenticate
$apiKey = authenticateAPI();
checkRateLimit($apiKey['id'], $apiKey['rate_limit'] ?? 1000);

$db = getDB();
$companyId = $apiKey['company_id'];

// GET: Logları listele
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $startDate = $_GET['start_date'] ?? date('Y-m-d', strtotime('-7 days'));
    $endDate = $_GET['end_date'] ?? date('Y-m-d');
    $afterId = (int)($_GET['after_id'] ?? 0);
    $limit = min((int)($_GET['limit'] ?? 100), 1000);
    $direction = $_GET['direction'] ?? null;
    $userId = $_GET['user_id'] ?? null;
    
    $sql = "
        SELECT al.id, al.employee_no, al.card_no, al.event_time, al.event_type,
               al.direction, al.verify_mode, al.status, al.temperature, al.mask_status,
               u.first_name, u.last_name,
               d.name as device_name, l.name as location_name
        FROM access_logs al
        LEFT JOIN users u ON al.user_id = u.id
        LEFT JOIN devices d ON al.device_id = d.id
        LEFT JOIN locations l ON d.location_id = l.id
        WHERE al.company_id = ?
        AND DATE(al.event_time) BETWEEN ? AND ?
    ";
    
    $params = [$companyId, $startDate, $endDate];
    
    if ($afterId > 0) {
        $sql .= " AND al.id > ?";
        $params[] = $afterId;
    }
    
    if ($direction) {
        $sql .= " AND al.direction = ?";
        $params[] = $direction;
    }
    
    if ($userId) {
        $sql .= " AND al.user_id = ?";
        $params[] = $userId;
    }
    
    $sql .= " ORDER BY al.id ASC LIMIT ?";
    $params[] = $limit;
    
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    $logs = $stmt->fetchAll();
    
    // Toplam sayı
    $stmt = $db->prepare("
        SELECT COUNT(*) FROM access_logs 
        WHERE company_id = ? AND DATE(event_time) BETWEEN ? AND ?
    ");
    $stmt->execute([$companyId, $startDate, $endDate]);
    $total = $stmt->fetchColumn();
    
    jsonSuccess([
        'data' => $logs,
        'total' => $total,
        'returned' => count($logs),
        'has_more' => count($logs) === $limit
    ]);
}

// POST: Senkronizasyon durumu bildir
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    $logIds = $input['log_ids'] ?? [];
    $status = $input['status'] ?? 'synced';
    
    if (empty($logIds)) {
        jsonError('log_ids gerekli', 400);
    }
    
    // Logları güncelle
    $placeholders = implode(',', array_fill(0, count($logIds), '?'));
    $stmt = $db->prepare("
        UPDATE access_logs 
        SET synced_to_pdks = 1, pdks_sync_time = NOW()
        WHERE company_id = ? AND id IN ({$placeholders})
    ");
    
    $params = array_merge([$companyId], $logIds);
    $stmt->execute($params);
    
    $affected = $stmt->rowCount();
    
    jsonSuccess([
        'synced_count' => $affected
    ], "{$affected} log senkronize edildi");
}

jsonError('Desteklenmeyen metod', 405);
